#pragma once

#include "model.h"

#include <solomon/services/memstore/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/clients/kikimr/rpc.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore {

class TStorageClusterEvents: private TEventSlot<EEventSpace::MemStore, ES_STORAGE_CLUSTER> {
    enum {
        GetTablets = SpaceBegin,
        ResolveTablets,
        Tablets,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Get all tables known in current cluster, in response cluster will send TTablets event.
     */
    struct TGetTablets: public NActors::TEventLocal<TGetTablets, GetTablets> {
    };

    /**
     * Resolve information about selected tablets, in response cluster will send TTablets event.
     */
    struct TResolveTablets: public NActors::TEventLocal<TResolveTablets, ResolveTablets> {
        std::vector<TTabletId> TabletIds;

        explicit TResolveTablets(std::vector<ui64> tabletIds) noexcept
            : TabletIds{std::move(tabletIds)}
        {
        }
    };

    /**
     * Requested information about tablets.
     */
    struct TTablets: public NActors::TEventLocal<TTablets, Tablets> {
        std::vector<TTabletActor> TabletActors;

        explicit TTablets(std::vector<TTabletActor> tabletActors) noexcept
            : TabletActors{std::move(tabletActors)}
        {
        }
    };
};

/**
 * Storage cluster options.
 */
struct TStorageClusterOptions {
    /**
     * Unique node identifier of current MemStore instance.
     */
    TNodeId NodeId{0};

    /**
     * Implementation of the RPC interface to communicate with Kikimr nodes.
     */
    std::shared_ptr<NKikimr::IKikimrClusterRpc> Rpc;

    /**
     * Path of solomon volume in Kikimr scheme which will be used as source of all known tablets.
     */
    TString VolumePath;

    /**
     * To find shard locations on nodes cluster will periodically get list of local
     * tablets from each node, this option configures delay between such operations.
     */
    TDuration WatchDelay = TDuration::Seconds(5);

    /**
     * Metric registry to report files size metrics
     */
    NMonitoring::TMetricRegistry& MetricRegistry;
};

/**
 * Creates actor for storage cluster.
 *
 * @param opts  cluster options
 * @return new instance of non registered actor
 */
std::unique_ptr<NActors::IActor> StorageCluster(TStorageClusterOptions opts);

} // namespace NSolomon::NMemStore
