#pragma once

#include <util/generic/string.h>
#include <util/string/printf.h>

#include <optional>

namespace NSolomon::NMemStore {

constexpr inline TStringBuf LOG_PREFIX = "c.m.l.";
constexpr inline TStringBuf TMP_LOG_PREFIX = "t.m.l.";
constexpr inline TStringBuf SNAPSHOT_PREFIX = "c.m.s.";
constexpr inline TStringBuf TMP_SNAPSHOT_PREFIX = "t.m.s.";
constexpr inline TStringBuf CURRENT_TXN_PREFIX = "c.m.txn.last.";
constexpr inline TStringBuf CURRENT_SNAPSHOT_STATE_PREFIX = "c.m.snapshot.last.";
constexpr ui16 NODE_ID_SIZE = 3;
constexpr ui32 MAX_NODE_ID = 1000; // pow(10, NODE_ID_SIZE)
constexpr ui16 TXN_SIZE = 17;
constexpr ui64 MAX_TXN = 100'000'000'000'000'000; // pow(10, TXN_SIZE)

enum class EFileType {
    Unknown,
    Log,
    TempLog,
    Snapshot,
    TempSnapshot
};

struct TFilenameParts {
    EFileType Type{EFileType::Unknown};
    // has value only if file type is known
    std::optional<ui32> NodeId;
};

TFilenameParts ParseMemstoreFilename(TStringBuf filename);

inline TString WalFilename(ui32 nodeId, ui64 txn, ui64 fileIdx, bool final, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.%017lu.%06lu%s",
            tmp ? TMP_LOG_PREFIX.data() : LOG_PREFIX.data(),
            nodeId,
            txn,
            fileIdx,
            final ? "z" : "y");
}

inline TString WalFilenamePrefix(ui32 nodeId, ui64 txn, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.%017lu.",
            tmp ? TMP_LOG_PREFIX.data() : LOG_PREFIX.data(),
            nodeId,
            txn);
}

inline TString WalFilenamePrefixShort(ui32 nodeId, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.",
            tmp ? TMP_LOG_PREFIX.data() : LOG_PREFIX.data(),
            nodeId);
}

inline TString WalTxnFilename(ui32 nodeId) {
    return Sprintf("%s%03u", CURRENT_TXN_PREFIX.data(), nodeId);
}

inline TString SnapshotFilename(ui32 nodeId, ui64 txn, ui64 fileIdx, bool final, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.%017lu.%06lu%s",
            tmp ? TMP_SNAPSHOT_PREFIX.data() : SNAPSHOT_PREFIX.data(),
            nodeId,
            txn,
            fileIdx,
            final ? "z" : "y");
}

inline TString SnapshotFilenamePrefix(ui32 nodeId, ui64 txn, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.%017lu.",
            tmp ? TMP_SNAPSHOT_PREFIX.data() : SNAPSHOT_PREFIX.data(),
            nodeId,
            txn);
}

inline TString SnapshotFilenamePrefixShort(ui32 nodeId, bool tmp) {
    Y_VERIFY(nodeId < MAX_NODE_ID, "node id %u > MAX_NODE_ID", nodeId);
    return Sprintf(
            "%s%03u.",
            tmp ? TMP_SNAPSHOT_PREFIX.data() : SNAPSHOT_PREFIX.data(),
            nodeId);
}

inline TString SnapshotTxnFilename(ui32 nodeId) {
    return Sprintf("%s%03u", CURRENT_SNAPSHOT_STATE_PREFIX.data(), nodeId);
}

} // namespace NSolomon::NMemStore
