#include "tablet.h"
#include "file_name.h"

#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/kv/kv_client.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore {
namespace {

constexpr TDuration ListFilesPeriod_ = TDuration::Minutes(5);
constexpr TDuration KvResponseMaxTime = TDuration::Seconds(60);

static_assert(KvResponseMaxTime < ListFilesPeriod_ / 2);

struct TLocalEvents: private TPrivateEvents {
    enum {
        ListFiles = SpaceBegin,
        ReceiveListFilesResponse,
        ReceiveListFilesError,
        End
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TListFiles: public NActors::TEventLocal<TListFiles, ListFiles> {
    };

    struct TReceiveListFilesResponse : public TEventLocal<TReceiveListFilesResponse, ReceiveListFilesResponse> {
        // all files in one group have the same prefix: log, snapshot or temp
        TVector<TVector<TFileInfo>> FileGroups;

        TReceiveListFilesResponse(TVector<TVector<TFileInfo>>&& filesInfo) noexcept
            : FileGroups(std::move(filesInfo))
        {
        }
    };

    struct TReceiveListFilesError: public TEventLocal<TReceiveListFilesError, ReceiveListFilesError> {
        TKvClientError Error;

        TReceiveListFilesError(TKvClientError&& error) noexcept
            : Error{std::move(error)}
        {
        }
    };
};

void SubscribeOnListFilesFuture(
        const TAsyncKvResult<TKikimrKvBatchResult>& listFilesFuture,
        TActorId sendTo,
        TActorSystem* actorSystem)
{
    listFilesFuture.Subscribe([sendTo, actorSystem](TAsyncKvResult<TKikimrKvBatchResult> result) {
        std::unique_ptr<IEventBase> event;
        try {
            auto valueOrError = result.ExtractValue();
            if (valueOrError.Success()) {
                event = std::make_unique<TLocalEvents::TReceiveListFilesResponse>(
                    std::move(valueOrError.Extract().ListFileQueryResults));
            } else {
                event = std::make_unique<TLocalEvents::TReceiveListFilesError>(valueOrError.ExtractError());
            }
        } catch (...) {
            event = std::make_unique<TLocalEvents::TReceiveListFilesError>(
                TKvClientError{grpc::StatusCode::UNKNOWN, CurrentExceptionMessage()});
        }
        actorSystem->Send(sendTo, event.release());
    });
}

EFileType GetFileGroupType(const TVector<TFileInfo>& fileGroup) {
    if (fileGroup.empty()) {
        return EFileType::Unknown;
    }

    auto fileNameParts = ParseMemstoreFilename(fileGroup[0].Name);
    return fileNameParts.Type;
}

ui64 CalcFileGroupSize(const TVector<TFileInfo>& fileGroup) {
    ui64 groupFilesSize{0};
    for (const auto& f: fileGroup) {
        groupFilesSize += f.SizeBytes;
    }
    return groupFilesSize;
}

class TStorageTablet: public TActorBootstrapped<TStorageTablet> {
public:
    TStorageTablet(
            TNodeId nodeId,
            TTabletId id,
            TString address,
            std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc) noexcept
        : NodeId_{nodeId}
        , Id_{id}
        , Address_{std::move(address)}
        , Rpc_{std::move(rpc)}
    {
        if (!Address_.Empty()) {
            NodeRpc_ = Rpc_->Get(Address_);
        }
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);
        Schedule(TFullJitter{}(ListFilesPeriod_), new TLocalEvents::TListFiles);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TStorageTabletEvents::TMoved, OnMoved)
            sFunc(TStorageTabletEvents::TLost, OnLost)
            hFunc(TStorageTabletEvents::TStatsReq, OnStats)
            hFunc(TStorageTabletEvents::TFilesSizeReq, OnCalcFilesSize)
            sFunc(TLocalEvents::TListFiles, OnListFiles)
            hFunc(TLocalEvents::TReceiveListFilesResponse, OnListFilesResponse)
            hFunc(TLocalEvents::TReceiveListFilesError, OnListFilesError)
            hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

    void OnMoved(const TStorageTabletEvents::TMoved::TPtr& ev) {
        Address_ = std::move(ev->Get()->Address);
        NodeRpc_ = Rpc_->Get(Address_);
    }

    void OnLost() {
        Address_.clear();
        NodeRpc_ = nullptr;
        FilesSizeInfo_.Clear();
    }

    void OnStats(const TStorageTabletEvents::TStatsReq::TPtr& ev) {
        auto* event = new TStorageTabletEvents::TStatsResp;
        event->TabletId = Id_;
        event->Address = Address_;
        event->ReadsRps = RandomNumber<double>();  // fake stats
        event->WritesRps = RandomNumber<double>(); // fake stats
        event->LogFilesSize = FilesSizeInfo_.LogFilesSize;
        event->SnapshotFilesSize = FilesSizeInfo_.SnapshotFilesSize;
        event->TempFilesSize = FilesSizeInfo_.TempFilesSize;
        Send(ev->Sender, event);
    }

    void OnSelfMon(const NSelfMon::TEvPageDataReq::TPtr& ev) {
        yandex::monitoring::selfmon::Page page;
        page.set_title("Storage tablet " + ToString(Id_));

        auto* g = page.mutable_grid();

        auto* r = g->add_rows();
        auto* obj = r->add_columns()->mutable_component()->mutable_object();
        {
            auto* f = obj->add_fields();
            f->set_name("Node Id");
            f->mutable_value()->set_uint64(NodeId_);
        }
        {
            auto* f = obj->add_fields();
            f->set_name("Tablet Id");
            f->mutable_value()->set_uint64(Id_);
        }
        {
            auto* f = obj->add_fields();
            f->set_name("Address");
            f->mutable_value()->set_string(Address_);
        }
        {
            auto* f = obj->add_fields();
            f->set_name("Log files size");
            f->mutable_value()->set_data_size(static_cast<double>(FilesSizeInfo_.LogFilesSize));
        }
        {
            auto* f = obj->add_fields();
            f->set_name("Snapshot files size");
            f->mutable_value()->set_data_size(static_cast<double>(FilesSizeInfo_.SnapshotFilesSize));
        }
        {
            auto* f = obj->add_fields();
            f->set_name("Temp files size");
            f->mutable_value()->set_data_size(static_cast<double>(FilesSizeInfo_.TempFilesSize));
        }

        Send(ev->Sender, new NSelfMon::TEvPageDataResp{std::move(page)});
    }

    void OnCalcFilesSize(const TStorageTabletEvents::TFilesSizeReq::TPtr& ev) {
        Send(ev->Sender, new TStorageTabletEvents::TFilesSizeResp(FilesSizeInfo_));
    }

    void OnListFiles() {
        if (NodeRpc_) {
            RequestFiles();
        } else {
            ScheduleListFilesMessage();
        }
    }

    void RequestFiles() {
        auto deadline = KvResponseMaxTime.ToDeadLine();
        auto batch = TKikimrKvBatchRequest(Id_, deadline);
        batch.ListPrefix(WalFilenamePrefixShort(NodeId_, false));
        batch.ListPrefix(SnapshotFilenamePrefixShort(NodeId_, false));
        batch.ListPrefix(WalFilenamePrefixShort(NodeId_, true));
        batch.ListPrefix(SnapshotFilenamePrefixShort(NodeId_, true));

        Y_ASSERT(NodeRpc_);
        TKikimrKvClient kvClient(NodeRpc_);
        auto batchResultFuture = kvClient.BatchRequest(batch);
        SubscribeOnListFilesFuture(batchResultFuture, SelfId(), TActorContext::ActorSystem());
    }

    void ScheduleListFilesMessage() {
        Schedule(ListFilesJitter_(ListFilesPeriod_), new TLocalEvents::TListFiles);
    }

    void OnListFilesResponse(const TLocalEvents::TReceiveListFilesResponse::TPtr& ev) {
        FilesSizeInfo_.Clear();
        for (const auto& fileGroup: ev->Get()->FileGroups) {
            const auto fileType = GetFileGroupType(fileGroup);
            switch (fileType) {
                case EFileType::Log:
                    FilesSizeInfo_.LogFilesSize += CalcFileGroupSize(fileGroup);
                    break;
                case EFileType::Snapshot:
                    FilesSizeInfo_.SnapshotFilesSize += CalcFileGroupSize(fileGroup);
                    break;
                case EFileType::TempLog:
                case EFileType::TempSnapshot:
                    FilesSizeInfo_.TempFilesSize += CalcFileGroupSize(fileGroup);
                    break;
                case EFileType::Unknown:
                    break;
            }
        }
        ScheduleListFilesMessage();
    }

    void OnListFilesError(TLocalEvents::TReceiveListFilesError::TPtr ev) {
        MON_WARN(StorageCluster, "Failed to list files on tablet " << Id_
            << ". Error: " << ev->Get()->Error.Message());
        ScheduleListFilesMessage();
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    const TNodeId NodeId_;
    const TTabletId Id_;
    TString Address_;
    std::shared_ptr<NKikimr::IKikimrClusterRpc> Rpc_;
    NKikimr::IKikimrRpc* NodeRpc_{nullptr};
    TTabletFilesSize FilesSizeInfo_;
    THalfJitter ListFilesJitter_;
};

} // namespace

std::unique_ptr<IActor> StorageTablet(
        TNodeId nodeId,
        TTabletId tabletId,
        TString address,
        std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc)
{
    return std::make_unique<TStorageTablet>(nodeId, tabletId, std::move(address), std::move(rpc));
}

} // namespace NSolomon::NMemStore
