#include <solomon/services/memstore/lib/storage/file_name.h>

#include <library/cpp/testing/gtest/gtest.h>

class TFileNameTest: public ::testing::Test {
protected:
    // The same as WalFilename but use any count of digits for nodeId
    static TString WalFilenameUnlimitedNodeId(ui32 nodeId, ui64 txn, ui64 fileIdx, bool final, bool tmp) {
        return Sprintf(
                "%s%03u.%017lu.%06lu%s",
                tmp ? NSolomon::NMemStore::TMP_LOG_PREFIX.data() : NSolomon::NMemStore::LOG_PREFIX.data(),
                nodeId,
                txn,
                fileIdx,
                final ? "z" : "y");
    }
};

using namespace NSolomon::NMemStore;

TEST_F(TFileNameTest, ParseFileType_Log) {
    const auto fileName = WalFilename(123, 456, 789, true, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Log);
}

TEST_F(TFileNameTest, ParseFileType_TempLog) {
    const auto fileName = WalFilename(123, 456, 789, true, true);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::TempLog);
}

TEST_F(TFileNameTest, ParseFileType_Snapsot) {
    const auto fileName = SnapshotFilename(123, 456, 789, true, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Snapshot);
}

TEST_F(TFileNameTest, ParseFileType_TempSnapsot) {
    const auto fileName = SnapshotFilename(123, 456, 789, true, true);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::TempSnapshot);
}

TEST_F(TFileNameTest, ParseNodeId_Zero) {
    const ui32 nodeId(0);
    const auto fileName = SnapshotFilename(nodeId, 456, 789, false, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Snapshot);
    ASSERT_EQ(fileNameParts.NodeId, nodeId);
}

TEST_F(TFileNameTest, ParseNodeId_Max) {
    const ui32 nodeId(MAX_NODE_ID - 1);
    const auto fileName = SnapshotFilename(nodeId, 456, 789, false, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Snapshot);
    ASSERT_EQ(fileNameParts.NodeId, nodeId);
}

TEST_F(TFileNameTest, TxnZero) {
    const ui32 nodeId(123);
    const ui64 txn(0);
    const auto fileName = SnapshotFilename(nodeId, txn, 789, false, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Snapshot);
    ASSERT_EQ(fileNameParts.NodeId, nodeId);
}

TEST_F(TFileNameTest, TxnMax) {
    const ui32 nodeId(832);
    const ui64 txn(MAX_TXN - 1);
    const auto fileName = WalFilename(nodeId, txn, 789, false, false);
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Log);
    ASSERT_EQ(fileNameParts.NodeId, nodeId);
}

TEST_F(TFileNameTest, ParseEmptyFileName) {
    const TString fileName;
    const auto fileNameParts = ParseMemstoreFilename(fileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Unknown);
    ASSERT_FALSE(fileNameParts.NodeId);
}

TEST_F(TFileNameTest, ParseBadFileName_BigNodeId) {
    ASSERT_EQ(WalFilename(123, 456, 789, true, false), WalFilenameUnlimitedNodeId(123, 456, 789, true, false));
    const auto badFileName = WalFilenameUnlimitedNodeId(1234, 456, 789, true, false);
    const auto fileNameParts = ParseMemstoreFilename(badFileName);
    ASSERT_EQ(fileNameParts.Type, EFileType::Unknown);
    ASSERT_FALSE(fileNameParts.NodeId);
}
