#pragma once

#include "model.h"

#include <solomon/services/memstore/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/clients/kikimr/rpc.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore {

class TStorageWatcherEvents: private TEventSlot<EEventSpace::DataProxy, ES_STORAGE_WATCHER> {
    enum {
        Subscribe = SpaceBegin,
        StateChanged,
        Resolve,
        ResolveResult,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Subscribe on tablets locations changes. Subscriber will receive changes on continuing basis.
     */
    struct TSubscribe: public NActors::TEventLocal<TSubscribe, Subscribe> {
    };

    /**
     * Latest changes in the location of tablets.
     */
    struct TStateChanged: public NActors::TEventLocal<TStateChanged, StateChanged> {
        /**
         * Locations of these tablets are changed or become known again.
         */
        std::vector<TTabletLocation> Moved;
        /**
         * Locations of these tablets become unknown.
         */
        std::vector<TTabletId> Lost;

        explicit TStateChanged(std::vector<TTabletLocation> moved, std::vector<TTabletId> lost) noexcept
            : Moved{std::move(moved)}
            , Lost{std::move(lost)}
        {
        }
    };

    /**
     * Get location of particular tablets.
     */
    struct TResolve: public NActors::TEventLocal<TResolve, Resolve> {
        /**
         * For which tablets locations are needed.
         */
        std::vector<TTabletId> Ids;

        explicit TResolve(std::vector<TTabletId> ids) noexcept
            : Ids{std::move(ids)}
        {
        }
    };

    /**
     * Locations of requested tablets.
     */
    struct TResolveResult: public NActors::TEventLocal<TResolveResult, ResolveResult> {
        /**
         * Only known locations are resturned, be ready for an empty vector.
         */
        std::vector<TTabletLocation> Locations;

        explicit TResolveResult(std::vector<TTabletLocation> locations) noexcept
            : Locations{std::move(locations)}
        {
        }
    };
};

/**
 * Create an actor which will watch KV tablets locations on Kikimr nodes.
 *
 * @param rpc           Kikimr cluster RPC implementation
 * @param volumePath    path of solomon volume to resolve tablets
 * @param updateDelay   minimal duration between updates
 * @return non registered actor instance
 */
std::unique_ptr<NActors::IActor> StorageClusterWatcher(
        std::shared_ptr<NKikimr::IKikimrClusterRpc> rpc,
        TString volumePath,
        TDuration updateDelay);

} // namespace NSolomon::NMemStore
