#pragma once

#include <solomon/services/memstore/lib/time_series/points_range.h>
#include <solomon/services/memstore/lib/time_series/time_series_frame.h>

#include <solomon/libs/cpp/ts_codec/bit_buffer.h>
#include <solomon/libs/cpp/ts_model/merge.h>
#include <solomon/libs/cpp/ts_model/point_traits.h>

#include <variant>

namespace NSolomon::NMemStore {

struct TFrameInfo {
    // size of frame class
    size_t FrameSizeBytes{0};
    // size of encoded points of frame
    size_t DataSizeBytes{0};
    // size of allocated buffer, containing encoded points
    size_t BufferSizeBytes{0};
    // size of all memory allocated by frame
    size_t TotalSizeBytes{0};
};

class IFrameHolder {
public:
    enum class EType: ui8 {
        Seal,
        SmallMutable,
        Mutable
    };

    virtual ~IFrameHolder() = default;
    virtual bool IsSealed() const = 0;
    virtual EType GetFrameType() const = 0;
    virtual std::unique_ptr<IFrameHolder> Seal() = 0;
    virtual void AddPoint(const NTs::TPointCommon& point, NTsModel::EPointType pointType) = 0;
    virtual TPointsRange Range() const = 0;
    virtual size_t NumPoints() const = 0;
    virtual bool IsEmpty() const = 0;
    virtual TTimeSeriesFrame ToTsFrame(NTsModel::EPointType type) = 0;
    virtual TFrameInfo GetFrameInfo() const = 0;
};

class TSealedFrame final : public IFrameHolder, TNonCopyable {
public:
    TSealedFrame() = default;
    ~TSealedFrame() override = default;

    TSealedFrame(NTs::TBitBuffer buffer, size_t numPoints, TPointsRange range) noexcept
            : Buffer_{std::move(buffer)}
            , NumPoints_{numPoints}
            , Range_{range}
    {
    }

    bool IsSealed() const override {
        return true;
    }

    EType GetFrameType() const override {
        return EType::Seal;
    }

    std::unique_ptr<IFrameHolder> Seal() override {
        Y_FAIL("Try to seal frame that already sealed");
        return {};
    }

    void AddPoint(const NTs::TPointCommon&, NTsModel::EPointType) override {
        Y_VERIFY(false, "Try to add point to a sealed time series frame");
    }

    TPointsRange Range() const override {
        return Range_;
    }

    size_t NumPoints() const override {
        return NumPoints_;
    }

    bool IsEmpty() const override {
        return NumPoints_ == 0;
    }

    TTimeSeriesFrame ToTsFrame(NTsModel::EPointType type) override {
        return TTimeSeriesFrame{
                Buffer_,     // copy
                NumPoints_,
                type,
                ~NTsModel::TPointColumns{},
                Range_};
    }

    TFrameInfo GetFrameInfo() const override {
        TFrameInfo info;
        info.FrameSizeBytes = sizeof(TSealedFrame);
        info.DataSizeBytes = Buffer_.Container().Size();
        info.BufferSizeBytes = Buffer_.CapacityBytes();
        info.TotalSizeBytes = info.FrameSizeBytes + info.BufferSizeBytes;
        return info;
    }

    [[nodiscard]] NTs::TBitBuffer TakeBuffer() {
        return std::move(Buffer_);
    }

private:
    NTs::TBitBuffer Buffer_;
    size_t NumPoints_{0};
    TPointsRange Range_;
};

template <typename TPoint>
class TMutableFrame final: public IFrameHolder, TNonCopyable {
public:
    TMutableFrame() = default;
    ~TMutableFrame() override = default;

    bool IsSealed() const override {
        return false;
    }

    EType GetFrameType() const override {
        return EType::Mutable;
    }

    explicit TMutableFrame(TBuffer&& buffer) noexcept
        : Buffer_{std::move(buffer)}
    {
    }

    std::unique_ptr<IFrameHolder> Seal() override {
        Sort();
        Encoder_.Flush();
        auto numPoints = Encoder_.FrameSize();
        return std::make_unique<TSealedFrame>(std::move(Buffer_), numPoints, Range_);
    }

    void AddPoint(const NTs::TPointCommon& pointCommon, NTsModel::EPointType pointType) override {
        Y_VERIFY(
                pointType == TPoint::Type,
                "Try to add point of wrong type %u to time series mutable frame. Expected type %u",
                static_cast<unsigned int>(pointType),
                static_cast<unsigned int>(TPoint::Type));

        const TPoint& point = static_cast<const TPoint&>(pointCommon);
        Encoder_.EncodePoint(point);

        if (point.Time < LastFramePoint_) {
            Sorted_ = false;
        } else if (point.Time == LastFramePoint_) {
            Merged_ = false;
        }

        LastFramePoint_ = point.Time;
        NumPoints_++;
        Range_.Update(point.Time);
    }

    TPointsRange Range() const override {
        return Range_;
    }

    size_t NumPoints() const override {
        return NumPoints_;
    }

    bool IsEmpty() const override {
        return LastFramePoint_ == TInstant::Zero();
    }

    TTimeSeriesFrame ToTsFrame(NTsModel::EPointType type) override {
        Y_VERIFY(type == TPoint::Type);
        Sort();
        Encoder_.Flush();
        auto numPoints = Encoder_.FrameSize();
        return TTimeSeriesFrame{
                Buffer_,   // copy
                numPoints,
                TPoint::Type,
                ~NTsModel::TPointColumns{},
                Range_};
    }

    TFrameInfo GetFrameInfo() const override {
        TFrameInfo info;
        info.FrameSizeBytes = sizeof(*this);
        info.DataSizeBytes = Buffer_.Container().Size();
        info.BufferSizeBytes = Buffer_.CapacityBytes();
        info.TotalSizeBytes = info.FrameSizeBytes + info.BufferSizeBytes;
        return info;
    }

private:
    void Sort() {
        if (!Sorted_) {
            DoSortAndMerge();
        } else if (!Merged_) {
            DoMerge();
        }
    }

    void DoSortAndMerge() {
        // TODO: separate this metric
        // TMetrics::Instance()->StorageReorderings->Inc();

        TVector<TPoint> unpacked;

        {
            Encoder_.Flush();
            auto decoder = NTsModel::TDecoder<TPoint>::Aggr(Buffer_);
            while (decoder.HasNext()) {
                auto point = &unpacked.emplace_back();
                decoder.NextPoint(point);
            }
        }

        StableSortBy(unpacked, [](const TPoint& point) { return point.Time; });

        if (!unpacked.empty()) {
            LastFramePoint_ = unpacked.back().Time;
        }

        Buffer_.Clear();
        Writer_.SetPos(0);
        Encoder_ = NTsModel::TEncoder<TPoint>::Aggr(&Writer_);

        for (auto it = unpacked.begin(); it != unpacked.end(); ++it) {
            while (it + 1 != unpacked.end() && it->Time == (it + 1)->Time) {
                NSolomon::NTsModel::Merge(*it, it + 1);
                it++;
                NumPoints_--;
            }
            Encoder_.EncodePoint(*it);
        }

        Sorted_ = true;
        Merged_ = true;
    }

    void DoMerge() {
        // TODO: separate this metric
        // TMetrics::Instance()->StorageMerges->Inc();

        Encoder_.Flush();
        auto bufferCopy = Buffer_;
        auto decoder = NTsModel::TDecoder<TPoint>::Aggr(bufferCopy);

        Buffer_.Clear();
        Writer_.SetPos(0);
        Encoder_ = NTsModel::TEncoder<TPoint>::Aggr(&Writer_);

        TPoint prevPoint, point;
        decoder.NextPoint(&prevPoint);

        while (decoder.NextPoint(&point)) {
            if (prevPoint.Time == point.Time) {
                NSolomon::NTsModel::Merge(prevPoint, &point);
                NumPoints_--;
            } else {
                Encoder_.EncodePoint(prevPoint);
            }
            std::swap(prevPoint, point);
        }

        Encoder_.EncodePoint(prevPoint);
        Merged_ = true;
    }

private:
    NTs::TBitBuffer Buffer_;
    NTs::TBitWriter Writer_{&Buffer_};
    NTsModel::TEncoder<TPoint> Encoder_{NTsModel::TEncoder<TPoint>::Aggr(&Writer_)};
    TInstant LastFramePoint_;
    TPointsRange Range_;
    size_t NumPoints_{0}; // estimated number, actual number will be available after encoder flush
    bool Sorted_{true};
    bool Merged_{true};
};

template <typename TPoint>
class TSmallMutableFrame final: public IFrameHolder, TNonCopyable {
public:
    TSmallMutableFrame() = default;
    ~TSmallMutableFrame() override = default;

    explicit TSmallMutableFrame(TBuffer&& buffer) noexcept
        : Buffer_{std::move(buffer)}
    {
    }

    bool IsSealed() const override {
        return false;
    }

    EType GetFrameType() const override {
        return EType::SmallMutable;
    }

    std::unique_ptr<IFrameHolder> Seal() override {
        return std::make_unique<TSealedFrame>(std::move(Buffer_), NumPoints_, Range_);
    }

    std::unique_ptr<IFrameHolder> CreateMutableFrame() {
        TVector<TPoint> unpacked = DecodePoints();
        auto frame = std::make_unique<TMutableFrame<TPoint>>();
        for (const auto& point: unpacked) {
            frame->AddPoint(point, TPoint::Type);
        }
        return frame;
    }

    void AddPoint(const NTs::TPointCommon& pointCommon, NTsModel::EPointType pointType) override {
        Y_VERIFY(
                pointType == TPoint::Type,
                "Try to add point of wrong type %u to time series small mutable frame. Expected type %u",
                static_cast<unsigned int>(pointType),
                static_cast<unsigned int>(TPoint::Type));
        Y_VERIFY(NumPoints_ < std::numeric_limits<decltype(NumPoints_)>::max(), "TSmallMutableFrame overflow");
        /**
         * Keep invariant: always store sorted and merged array of points
         */
        const TPoint& point = static_cast<const TPoint&>(pointCommon);
        TVector<TPoint> unpacked = DecodePoints();
        auto ins = std::upper_bound(unpacked.begin(), unpacked.end(), point, [](const TPoint& p1, const TPoint& p2) {
            return p1.Time < p2.Time;
        });
        bool isMerged = false;
        if (ins != unpacked.begin()) {
            TPoint& prevPoint = *(ins - 1);
            if (prevPoint.Time == point.Time) {
                TPoint prevCopy(prevPoint);
                prevPoint = point;
                NSolomon::NTsModel::Merge(prevCopy, &prevPoint);
                isMerged = true;
            }
        }
        if (!isMerged) {
            unpacked.insert(ins, point);
            NumPoints_++;
        }
        EncodePoints(unpacked);
        Range_.Update(TPointsRange{unpacked.front().Time, unpacked.back().Time});
    }

    TPointsRange Range() const override {
        return Range_;
    }

    size_t NumPoints() const override {
        return static_cast<size_t>(NumPoints_);
    }

    bool IsEmpty() const override {
        return NumPoints_ == 0;
    }

    TTimeSeriesFrame ToTsFrame(NTsModel::EPointType type) override {
        Y_VERIFY(type == TPoint::Type);
        return TTimeSeriesFrame{
                Buffer_,   // copy
                static_cast<size_t>(NumPoints_),
                TPoint::Type,
                ~NTsModel::TPointColumns{},
                Range_};
    }

    TFrameInfo GetFrameInfo() const override {
        TFrameInfo info;
        info.FrameSizeBytes = sizeof(*this);
        info.DataSizeBytes = Buffer_.Container().Size();
        info.BufferSizeBytes = Buffer_.CapacityBytes();
        info.TotalSizeBytes = info.FrameSizeBytes + info.BufferSizeBytes;
        return info;
    }

private:
    TVector<TPoint> DecodePoints() const {
        if (NumPoints_ == 0) {
            return {};
        }
        // + 1 because this method is called before insertion of a new point
        TVector<TPoint> points(::Reserve(static_cast<size_t>(NumPoints_) + 1));
        points.resize(static_cast<size_t>(NumPoints_));
        auto decoder = NTsModel::TDecoder<TPoint>::Aggr(Buffer_);
        for (TPoint& point: points) {
            decoder.NextPoint(&point);
        }
        return points;
    }

    void EncodePoints(const TVector<TPoint>& points) {
        Buffer_.Clear();
        NTs::TBitWriter writer{&Buffer_};
        NTsModel::TEncoder<TPoint> encoder{NTsModel::TEncoder<TPoint>::Aggr(&writer)};
        for (const auto& point: points) {
            encoder.EncodePoint(point);
        }
        encoder.Flush();
        writer.Flush();
    }

private:
    NTs::TBitBuffer Buffer_;
    TPointsRange Range_;
    size_t NumPoints_{0};
};

} // namespace NSolomon::NMemStore
