#include "time_series.h"
#include "frame.h"

#include <solomon/libs/cpp/ts_model/merge.h>
#include <solomon/libs/cpp/ts_model/iterator_crop.h>
#include <solomon/libs/cpp/ts_model/loghist_to_hist_iterator.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/deque.h>
#include <util/generic/list.h>
#include <util/generic/scope.h>
#include <util/string/cast.h>

using NMonitoring::EMetricValueType;

namespace NSolomon::NMemStore {
namespace {
/**
 * Buffer pool size is limited.
 * If unlimited it leads to excessive memory consumption up to 15% of total time series size.
 * https://st.yandex-team.ru/SOLOMON-8503
 */
const size_t MAX_BUFFER_POOL_SIZE = 0; // now pool is actually off

template <typename TPoint>
class TTimeSeriesImpl final: public TTimeSeries {
public:
    TTimeSeriesImpl(TFrameIdx begin, TFrameIdx seal, TFrameIdx end)
        : FirstFrame_{begin}
    {
        for (auto frame = begin; frame < seal; ++frame) {
            Frames_.emplace_back(std::make_unique<TSealedFrame>());
        }
        for (auto frame = seal; frame < end; ++frame) {
            Frames_.emplace_back(std::make_unique<TSmallMutableFrame<TPoint>>(std::move(GetBuffer())));
        }
    }

public:
    TTimeSeriesImpl(TFrameIdx begin, TVector<TTimeSeriesFrame>&& frames)
        : FirstFrame_{begin}
    {
        auto curFrame = FirstFrame_;
        for (auto& frame: frames) {
            AddFrame(curFrame);
            auto logHistIter = frame.Iterator();
            Y_VERIFY(logHistIter->Type() == NTsModel::EPointType::LogHist,
                     "Unavailable type of cast. Cannot convert from %s", ToString(logHistIter->Type()).c_str());
            Y_VERIFY(TPoint::Type == NTsModel::EPointType::Hist,
                     "Unavailable type of cast. Cannot convert to %s", ToString(TPoint::Type).c_str());
            NTsModel::TLogHistToHistIterator logHistToHistIter(logHistIter->As<NTsModel::TLogHistPoint>());
            AddPoint(curFrame, &logHistToHistIter);
            curFrame++;
        }
    }

private:
    TTimeSeriesFrame ReadFrame(IFrameHolder* frame) {
        Y_VERIFY(frame, "try to read null time series frame pointer");
        return frame->ToTsFrame(TPoint::Type);
    }

    bool FrameContains(const IFrameHolder* frame, TInstant min, TInstant max) {
        return frame->Range().Intersects(min, max);
    }

private:
    const std::unique_ptr<IFrameHolder>& GetFrameHolder(TFrameIdx frameIdx) const {
        Y_VERIFY(frameIdx >= FrameIdxBegin() && frameIdx < FrameIdxEnd(), "unknown frame index %ld", frameIdx);
        auto& frameHolder = Frames_[frameIdx - FrameIdxBegin()];
        Y_VERIFY(frameHolder.get() != nullptr, "time series frame holder with index %ld has a null pointer", frameIdx);
        return frameHolder;
    }

    std::unique_ptr<IFrameHolder>& GetFrameHolder(TFrameIdx frameIdx) {
        Y_VERIFY(frameIdx >= FrameIdxBegin() && frameIdx < FrameIdxEnd(), "unknown frame index %ld", frameIdx);
        auto& frameHolder = Frames_[frameIdx - FrameIdxBegin()];
        Y_VERIFY(frameHolder.get() != nullptr, "time series frame holder with index %ld has a null pointer", frameIdx);
        return frameHolder;
    }

    const IFrameHolder* GetFrame(TFrameIdx frameIdx) const {
        return GetFrameHolder(frameIdx).get();
    }

    IFrameHolder* GetFrame(TFrameIdx frameIdx) {
        return GetFrameHolder(frameIdx).get();
    }

    const IFrameHolder* GetMutableFrame(TFrameIdx frameIdx) const {
        auto& frameHolder = GetFrameHolder(frameIdx);
        return frameHolder->IsSealed() ? nullptr : frameHolder.get();
    }

    IFrameHolder* GetMutableFrame(TFrameIdx frameIdx) {
        auto& frameHolder = GetFrameHolder(frameIdx);
        return frameHolder->IsSealed() ? nullptr : frameHolder.get();
    }

public:
    NTsModel::EPointType MetricType() const override {
        return TPoint::Type;
    }

    bool Empty() const override {
        return NumPoints() == 0;
    }

    size_t NumPoints() const override {
        size_t numPoints = 0;
        for (const auto& frame: Frames_) {
            numPoints += frame->NumPoints();
        }
        return numPoints;
    }

    size_t NumFrames() const override {
        return Frames_.size();
    }

    TFrameIdx FrameIdxBegin() const override {
        return FirstFrame_;
    }

    TFrameIdx FrameIdxEnd() const override {
        return FirstFrame_ + Frames_.size();
    }

    TPointsRange AddPoint(TFrameIdx frameIdx, NSLog::TTsParser& data) override {
        if (!Convertible(data.GetMetricType(), MetricType())) {
            ythrow TTypeError() << "unable to convert " << data.GetMetricType() << " to " << MetricType();
        }
        auto itr = data.Iterator<TPoint>();
        return AddPoint(frameIdx, &itr);
    }

    TPointsRange AddPoint(TFrameIdx frameIdx, NTsModel::IGenericIterator* data) {
        auto& frameHolder = GetFrameHolder(frameIdx);
        Y_VERIFY(!frameHolder->IsSealed(), "adding a point to a sealed frame [%ud] ", static_cast<ui32>(frameIdx));
        if (frameHolder->GetFrameType() == IFrameHolder::EType::SmallMutable && frameHolder->NumPoints() >= 2) {
            auto* smallFrame = dynamic_cast<TSmallMutableFrame<TPoint>*>(frameHolder.get());
            Y_VERIFY(smallFrame);
            frameHolder = smallFrame->CreateMutableFrame();
        }
        auto* frameMut = frameHolder.get();
        NTsModel::IIterator<TPoint>* iter = data->As<TPoint>();
        TPointsRange range;
        for (TPoint point; iter->NextPoint(&point);) {
            frameMut->AddPoint(point, MetricType());
            range.Update(point.Time);
        }
        return range;
    }

    void AddFrame(TFrameIdx frameIdx) override {
        Y_VERIFY(FirstFrame_ + Frames_.size() == frameIdx, "frame idx mismatch on add. frame id: %lu, expected: %lu",
                 frameIdx, FirstFrame_+ Frames_.size());
        Frames_.emplace_back(std::make_unique<TSmallMutableFrame<TPoint>>(std::move(GetBuffer())));
    }

    void SealFrame(TFrameIdx frameIdx) override {
        auto* frameMut = GetMutableFrame(frameIdx);
        Y_VERIFY(frameMut, "sealing an already sealed frame");
        Frames_[frameIdx - FrameIdxBegin()] = frameMut->Seal();
    }

    void DropFrame(TFrameIdx frameIdx) override {
        if (Frames_.empty()) {
            // no frame to drop
            return;
        }

        Y_VERIFY(FirstFrame_ == frameIdx, "frame idx mismatch on drop. frame id: %lu, expected: %lu", frameIdx, FirstFrame_);

        auto* frame = Frames_.front().get();
        Y_VERIFY(frame, "dropping a frame that already is dropped");
        Y_VERIFY(frame->IsSealed(), "dropping a non-sealed frame");

        if (BufferPool_.size() < MAX_BUFFER_POOL_SIZE) {
            auto* sealFrame = static_cast<TSealedFrame*>(frame);
            BufferPool_.push_back(sealFrame->TakeBuffer().Container());
        }
        Frames_.erase(Frames_.begin());
        FirstFrame_++;
    }

    TTimeSeriesFrame ReadFrame(TFrameIdx frameIdx) override {
        return ReadFrame(GetFrame(frameIdx));
    }

    TTimeSeriesFrames ReadBetween(TInstant begin, TInstant end) override {
        TVector<TTimeSeriesFrame> result;
        for (auto& frame: Frames_) {
            if (FrameContains(frame.get(), begin, end)) {
                result.push_back(ReadFrame(frame.get()));
            }
        }
        return TTimeSeriesFrames{MetricType(), ~NTsModel::TPointColumns{}, std::move(result), begin, end};
    }

    static bool Convertible(NTsModel::EPointType from, NTsModel::EPointType to) {
        if (from == NTsModel::EPointType::LogHist && to == NTsModel::EPointType::Hist) {
            return true;
        }
        return from == to;
    }

    TBuffer GetBuffer() {
        TBuffer buf;
        if (!BufferPool_.empty()) {
            buf = std::move(*BufferPool_.begin());
            buf.Clear();
            BufferPool_.erase(BufferPool_.begin());
        }
        return std::move(buf);
    }

    bool NeedsToConvert(NTsModel::EPointType pointType) override {
        // only one avalible convertion:
        return MetricType() == NTsModel::EPointType::LogHist && pointType == NTsModel::EPointType::Hist;
    }

    TIntrusivePtr<TTimeSeries> ConvertTo(NTsModel::EPointType pointType) override {
        Y_VERIFY(MetricType() != pointType, "No needs to convert TimeSeries. Metric and data have same type: %s",
                 ToString(MetricType()).c_str());
        Y_VERIFY(NeedsToConvert(pointType), "Cannot convert TimeSeries from %s to %s",
                 ToString(MetricType()).c_str(), ToString(pointType).c_str());
        TVector<TTimeSeriesFrame> readFrames;

        for (auto& frame: Frames_) {
            TTimeSeriesFrame ts_frame = ReadFrame(frame.get());
            readFrames.push_back(std::move(ts_frame));
        }
        auto newTs = NTsModel::Visit(pointType, [&](auto traits) -> TIntrusivePtr<TTimeSeries> {
            using TDataPoint = typename decltype(traits)::TPoint;
            return TIntrusivePtr<TTimeSeriesImpl<TDataPoint>>(
                new TTimeSeriesImpl<TDataPoint>(FirstFrame_, std::move(readFrames)));
        });
        auto curFrame = FirstFrame_;
        for (auto& frame: Frames_) {
            if (frame->IsSealed()) {
                newTs->SealFrame(curFrame);
            }
            curFrame++;
        }
        return newTs;
    }

    bool LastFrameIsEmpty() const override {
        return Frames_.back()->IsEmpty();
    }

    void Compact() override {
        Frames_.shrink_to_fit();
    }

    size_t SizeBytes() const override {
        size_t size = sizeof(*this);

        for (const auto& frame: Frames_) {
            size += frame->GetFrameInfo().TotalSizeBytes;
        }

        for (const auto& buffer: BufferPool_) {
            size += sizeof(buffer) + buffer.Capacity();
        }

        return size;
    }

    TTsMemoryStat CalcMemoryStat() const override {
        TTsMemoryStat stat;
        for (const auto& frame: Frames_) {
            const bool isSealed = frame->IsSealed();
            const TFrameInfo info = frame->GetFrameInfo();
            TFramesMemoryStat& frameStat = isSealed ? stat.SealedFramesStat : stat.MutableFramesStat;
            frameStat.FramesCount++;
            frameStat.PointsCount += frame->NumPoints();
            frameStat.DataSizeBytes += info.DataSizeBytes;
            frameStat.BuffersSizeBytes += info.BufferSizeBytes;
            frameStat.FramesSizeBytes += info.FrameSizeBytes;
            frameStat.FrameHoldersSizeBytes += sizeof(frame);
        }

        for (const auto& buffer: BufferPool_) {
            stat.BuffersCacheSizeBytes += sizeof(buffer) + buffer.Capacity();
        }

        stat.DequeCapacityBytes = Frames_.capacity() * sizeof(std::unique_ptr<IFrameHolder>);

        return stat;
    }

private:
    TVector<std::unique_ptr<IFrameHolder>> Frames_;
    TFrameIdx FirstFrame_{0};
    TList<TBuffer> BufferPool_;
};

} // namespace

TIntrusivePtr<TTimeSeries> TTimeSeries::Create(NTsModel::EPointType type, TFrameIdx begin, TFrameIdx seal, TFrameIdx end) {
    return NTsModel::Visit(type, [&](auto traits) -> TIntrusivePtr<TTimeSeries> {
        using TPoint = typename decltype(traits)::TPoint;
        return MakeIntrusive<TTimeSeriesImpl<TPoint>>(begin, seal, end);
    });
}

} // namespace NSolomon::NMemStore
