#pragma once

#include "points_range.h"
#include "time_series_frame.h"
#include "time_series_frames.h"
#include "ts_memory_stat.h"

#include <solomon/services/memstore/lib/types.h>
#include <solomon/services/memstore/lib/slog/parser.h>
#include <solomon/libs/cpp/selectors/selectors.h>
#include <solomon/libs/cpp/ts_codec/counter_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/gauge_int_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_log_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/summary_double_ts_codec.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/system/types.h>
#include <util/memory/blob.h>

namespace NSolomon::NMemStore {

class TTypeError: public yexception{
};

/**
 * Optimized data storage for time series.
 */
class TTimeSeries: public TThrRefBase {
public:
    /**
     * Get type of data in this series.
     */
    virtual NTsModel::EPointType MetricType() const = 0;

    /**
     * Check if this time series contains no points.
     */
    virtual bool Empty() const = 0;

    /**
     * Get number of points in this time series.
     */
    virtual size_t NumPoints() const = 0;

    /**
     * Get number of frames that're currently stored in this time series.
     */
    virtual size_t NumFrames() const = 0;

    /**
     * Get index of the first frame stored in this time series.
     */
    virtual TFrameIdx FrameIdxBegin() const = 0;

    /**
     * Get index of the past-the-last frame stored in this time series.
     */
    virtual TFrameIdx FrameIdxEnd() const = 0;

    /**
     * Add new point into the given frame. This method panics if the frame is sealed or deleted.
     * Also panics if type of new data does not match type of the time series.
     * @returns min and max timestamps from data
     */
    virtual TPointsRange AddPoint(TFrameIdx frameIdx, NSLog::TTsParser& data) = 0;

    /**
     * Add a new frame to the time series. Frame index should be zero for the first frame, and incremental for all
     * consecutive frames. It is used for debug checks to ensure frames being consistent across subshards.
     */
    virtual void AddFrame(TFrameIdx frameIdx) = 0;

    /**
     * Mark frame with the given index as sealed. Sealed frames are immutable and guaranteed to be sorted.
     */
    virtual void SealFrame(TFrameIdx frameIdx) = 0;

    /**
     * Delete the first frame from this time series. Panics if there are no frames or if the frame is not sealed.
     * Frame index should be equal to the index of the frame that's about to be dropped. It is used for debug
     * checks to ensure frames being consistent across subshards.
     */
    virtual void DropFrame(TFrameIdx frameIdx) = 0;

    /**
     * Read frame data. If frame is sealed, returns a pointer to the frame data. If frame is not sealed,
     * sorts it if necessary and returns a copy of frame's data. In either case the returned data is not affected
     * by calls to `AddPoint`, and is guaranteed to stay valid even after this class dies.
     */
    virtual TTimeSeriesFrame ReadFrame(TFrameIdx frameIdx) = 0;

    /**
     * Read all frames that contain at least one point from the given time interval.
     *
     * Frame data is not processed, so returned frames may contain points not from the interval.
     */
    virtual TTimeSeriesFrames ReadBetween(TInstant windowBegin, TInstant windowEnd) = 0;

    /**
     * Convert TimeSeries to type witch support data
     */
    virtual TIntrusivePtr<TTimeSeries> ConvertTo(NTsModel::EPointType) = 0;

    /**
     * Chech if time series needs convertion for future data processing
     */
    virtual bool NeedsToConvert(NTsModel::EPointType) = 0;

    /**
     * Check if last frame have no points
     */
     virtual bool LastFrameIsEmpty() const = 0;

     virtual size_t SizeBytes() const = 0;

     virtual TTsMemoryStat CalcMemoryStat() const = 0;

     virtual void Compact() = 0;

public:
    /**
     * Create storage for metric of the given type.
     */
    static TIntrusivePtr<TTimeSeries> Create(NTsModel::EPointType, TFrameIdx begin, TFrameIdx seal, TFrameIdx end);
};

using TTimeSeriesPtr = TIntrusivePtr<TTimeSeries>;

} // NSolomon::NMemStore
