#pragma once
#include "time_series_frame.h"

#include <solomon/libs/cpp/ts_codec/bit_buffer.h>
#include <solomon/libs/cpp/ts_model/iterable.h>

#include <util/generic/array_ref.h>
#include <util/generic/vector.h>

namespace NSolomon::NMemStore {
/**
* Multiple frames containing points between given timestamps.
*/
class TTimeSeriesFrames final: public NTsModel::IIterable {
public:
    TTimeSeriesFrames(
            NTsModel::EPointType metricType,
            NTsModel::TPointColumns columns,
            TVector<TTimeSeriesFrame> frames,
            TInstant windowBegin,
            TInstant windowEnd)
        : MetricType_{metricType}
        , Columns_{columns}
        , Frames_(std::move(frames))
        , WindowBegin_{windowBegin}
        , WindowEnd_{windowEnd}
    {
        for (auto& frame: Frames_) {
            Y_ASSERT(frame.Type() == MetricType_);
        }
    }

public:
    /**
     * Get all frames (with untrimmed data in them).
     */
    TConstArrayRef<TTimeSeriesFrame> Frames() const noexcept {
        return Frames_;
    }

    /**
     * Get type of this frame.
     */
    NTsModel::EPointType Type() const override {
        return MetricType_;
    }

    /**
     * Get ts columns for this data.
     */
    NTsModel::TPointColumns Columns() const override {
        return Columns_;
    }

    /**
     * Read data from frames: merge them and trim the result to only include points between given timestamps.
     */
    THolder<NTsModel::IGenericIterator> Iterator() const override;

    /**
     * Beginning of the time window that constraints points in this iterable.
     */
    TInstant WindowBegin() const override {
        return WindowBegin_;
    }

    /**
     * Ending of the time window that constraints points in this iterable.
     */
    TInstant WindowEnd() const override {
        return WindowEnd_;
    }

private:
    NTsModel::EPointType MetricType_;
    NTsModel::TPointColumns Columns_;
    TVector<TTimeSeriesFrame> Frames_;
    TInstant WindowBegin_;
    TInstant WindowEnd_;
};

} // namespace NSolomon::NMemStore
