#include <solomon/services/memstore/lib/time_series/time_series.h>
#include <solomon/services/memstore/lib/index/metrics.h>

#include <solomon/libs/cpp/slog/log_data.h>
#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/testlib/testlib.h>
#include <solomon/libs/cpp/ts_model/points.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon::NMemStore;

void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TGaugePoint* point) {
    *point = NSolomon::NTsModel::Gauge(ts, val, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TCounterPoint* point) {
    *point = NSolomon::NTsModel::Counter(ts, val, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TRatePoint* point) {
    *point = NSolomon::NTsModel::Rate(ts, val, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TIGaugePoint* point) {
    *point = NSolomon::NTsModel::IGauge(ts, val, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::THistPoint* point) {
    *point = NSolomon::NTsModel::Hist(ts, {{1, static_cast<ui64>(val)}, {2, 0}}, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::THistRatePoint* point) {
    *point = NSolomon::NTsModel::HistRate(ts, {{1, static_cast<ui64>(val)}, {2, 0}}, 0);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TDSummaryPoint* point) {
    *point = NSolomon::NTsModel::DSummary(ts, 1, val, val, val, val);
}
void MakeReferenceValue(TStringBuf ts, ui32 val, NSolomon::NTsModel::TLogHistPoint* point) {
    *point = NSolomon::NTsModel::LogHist(ts, 0, 2, {static_cast<double>(val), 0});
}

template <typename TIterable>
void ExpectSeriesEq(
        const TIterable& iterable,
        std::initializer_list<std::pair<TStringBuf, ui32>> data)
{
    auto begin = data.begin();
    auto end = data.end();

    size_t i = 0;

    NSolomon::NTsModel::Visit(iterable.Type(), [&](auto traits) {
        auto genericIt = iterable.Iterator();
        auto& it = traits.DowncastIterator(*genericIt);
        auto got = traits.MakePoint();
        auto hasPoint = false;
        auto expected = traits.MakePoint();
        while ((hasPoint = it.NextPoint(&got)) && begin != end) {
            MakeReferenceValue(begin->first, begin->second, &expected);
            EXPECT_EQ(got, expected) << "point #" << i;
            begin++;
            i++;
        }

        if (hasPoint) {
            ADD_FAILURE() << "frame have more values than expected, first extra value is " << got;
        }

        if (begin != end) {
            MakeReferenceValue(begin->first, begin->second, &expected);
            ADD_FAILURE() << "frame have less values than expected, first missing value is " << expected;
        }
    });
}

class TimeSeries: public ::testing::Test, public testing::WithParamInterface<NSolomon::NTsModel::EPointType> {
protected:
    virtual void SetUp() {
        Registry = std::make_shared<NMonitoring::TMetricRegistry>();
        Metrics = std::make_shared<NSolomon::NMemStore::NIndex::TMetrics>(Registry);
    }

public:
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry;
    std::shared_ptr<NSolomon::NMemStore::NIndex::TMetrics> Metrics;
};

NSolomon::NMemStore::NSLog::TTsParser Point(NSolomon::NSlog::TDataPoint&& point) {
    auto& [meta, data] = point;
    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};
    return parser.Next();
}

TEST_P(TimeSeries, Create) {
    {
        auto ts = TTimeSeries::Create(GetParam(), 0, 0, 0);
        EXPECT_EQ(ts->MetricType(), GetParam());
        EXPECT_TRUE(ts->Empty());
        EXPECT_EQ(ts->NumPoints(), 0u);
        EXPECT_EQ(ts->NumFrames(), 0u);
        EXPECT_EQ(ts->FrameIdxBegin(), 0u);
        EXPECT_EQ(ts->FrameIdxEnd(), 0u);
    }
}

TEST_P(TimeSeries, CreateNonEmptyFrames) {
    {
        auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);
        EXPECT_EQ(ts->MetricType(), GetParam());
        EXPECT_TRUE(ts->Empty());
        EXPECT_EQ(ts->NumPoints(), 0u);
        EXPECT_EQ(ts->NumFrames(), 1u);
        EXPECT_EQ(ts->FrameIdxBegin(), 0u);
        EXPECT_EQ(ts->FrameIdxEnd(), 1u);
    }

    {
        auto ts = TTimeSeries::Create(GetParam(), 2, 4, 5);
        EXPECT_EQ(ts->MetricType(), GetParam());
        EXPECT_TRUE(ts->Empty());
        EXPECT_EQ(ts->NumPoints(), 0u);
        EXPECT_EQ(ts->NumFrames(), 3u);
        EXPECT_EQ(ts->FrameIdxBegin(), 2u);
        EXPECT_EQ(ts->FrameIdxEnd(), 5u);
    }
}

TEST_P(TimeSeries, AddPoint) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 5)
                    .Add("2000-01-01T00:00:15Z", 10)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 2u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:15Z", 10}});
    }

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:30Z", 5)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 3u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:15Z", 10}, {"2000-01-01T00:00:30Z", 5}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);
}

TEST_P(TimeSeries, AddPointUnsorted) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:15Z", 5)
                    .Add("2000-01-01T00:00:10Z", 10)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 2u);

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
     EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:10Z", 10}, {"2000-01-01T00:00:15Z", 5}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:10Z", 10}, {"2000-01-01T00:00:15Z", 5}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);
}

TEST_P(TimeSeries, AddPointUnsortedUnmerged) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:10Z", 10)
                    .Add("2000-01-01T00:00:00Z", 0)
                    .Add("2000-01-01T00:00:10Z", 15)
                    .Add("2000-01-01T00:00:20Z", 20)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:00Z", 0}, {"2000-01-01T00:00:10Z", 15}, {"2000-01-01T00:00:20Z", 20}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);
}

TEST_P(TimeSeries, AddPointUnmerged) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 0)
                    .Add("2000-01-01T00:00:10Z", 10)
                    .Add("2000-01-01T00:00:10Z", 15)
                    .Add("2000-01-01T00:00:20Z", 20)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(0),
            {{"2000-01-01T00:00:00Z", 0}, {"2000-01-01T00:00:10Z", 15}, {"2000-01-01T00:00:20Z", 20}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 1u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);
}

TEST_P(TimeSeries, AddPointMiddleFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 3);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 5)
                    .Add("2000-01-01T00:00:15Z", 10)
                    .Done()
                .Done());
        ts->AddPoint(1, data);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 2u);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 5)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 3u);
}

TEST_P(TimeSeries, AddPointSealedFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 1, 2);

    auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:15Z", 10)
                .Done()
            .Done());
    EXPECT_DEATH(
        ts->AddPoint(0, data),
        "adding a point to a sealed frame");
}

TEST_P(TimeSeries, AddPointNonExistentFrameAhead) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
    EXPECT_DEATH(
        ts->AddPoint(2, data),
        "unknown frame index 2");
}

TEST_P(TimeSeries, AddPointNonExistentFrameBehind) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
    EXPECT_DEATH(
        ts->AddPoint(0, data),
        "unknown frame index 0");
}

TEST_P(TimeSeries, AddFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 1);

    EXPECT_EQ(ts->NumFrames(), 0u);
    EXPECT_EQ(ts->FrameIdxBegin(), 1u);
    EXPECT_EQ(ts->FrameIdxEnd(), 1u);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        EXPECT_DEATH(
            ts->AddPoint(1, data),
            "unknown frame index 1");
    }

    ts->AddFrame(1);

    EXPECT_EQ(ts->NumFrames(), 1u);
    EXPECT_EQ(ts->FrameIdxBegin(), 1u);
    EXPECT_EQ(ts->FrameIdxEnd(), 2u);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    {
        ExpectSeriesEq(
            ts->ReadFrame(1),
            {{"2000-01-01T00:00:00Z", 5}});
    }
}

TEST_P(TimeSeries, AddFrameIdxMismatch) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 1);

    EXPECT_DEATH(
        ts->AddFrame(0),
        "frame idx mismatch on add");

    EXPECT_DEATH(
        ts->AddFrame(2),
        "frame idx mismatch on add");
}

TEST_P(TimeSeries, SealFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    ts->SealFrame(1);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        EXPECT_DEATH(
            ts->AddPoint(1, data),
            "adding a point to a sealed frame");
    }

    {
        ExpectSeriesEq(
            ts->ReadFrame(1),
            {{"2000-01-01T00:00:00Z", 5}});
    }
}

TEST_P(TimeSeries, SealFrameUnsorted) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:15Z", 15)
                .Add("2000-01-01T00:00:00Z", 0)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    ts->SealFrame(1);

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        EXPECT_DEATH(
            ts->AddPoint(1, data),
            "adding a point to a sealed frame");
    }

    {
        ExpectSeriesEq(
            ts->ReadFrame(1),
            {{"2000-01-01T00:00:00Z", 0}, {"2000-01-01T00:00:15Z", 15}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);
}

TEST_P(TimeSeries, SealFrameAlreadySealed) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:15Z", 15)
                .Add("2000-01-01T00:00:00Z", 0)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(1),
            {{"2000-01-01T00:00:00Z", 0}, {"2000-01-01T00:00:15Z", 15}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    ts->SealFrame(1);

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    {
        ExpectSeriesEq(
            ts->ReadFrame(1),
            {{"2000-01-01T00:00:00Z", 0}, {"2000-01-01T00:00:15Z", 15}});
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);
}

TEST_P(TimeSeries, SealFrameNonExistentFrameAhead) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    EXPECT_DEATH(
        ts->SealFrame(2),
        "unknown frame index 2");
}

TEST_P(TimeSeries, SealFrameNonExistentFrameBehind) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    EXPECT_DEATH(
        ts->SealFrame(0),
        "unknown frame index 0");
}

TEST_P(TimeSeries, DropFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 1);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Done()
            .Done());
        ts->AddPoint(0, data);
        ts->SealFrame(0);
    }

    EXPECT_FALSE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 2u);
    EXPECT_EQ(ts->NumFrames(), 1u);
    EXPECT_EQ(ts->FrameIdxBegin(), 0u);
    EXPECT_EQ(ts->FrameIdxEnd(), 1u);

    ts->DropFrame(0);

    EXPECT_TRUE(ts->Empty());
    EXPECT_EQ(ts->NumPoints(), 0u);
    EXPECT_EQ(ts->NumFrames(), 0u);
    EXPECT_EQ(ts->FrameIdxBegin(), 1u);
    EXPECT_EQ(ts->FrameIdxEnd(), 1u);
}

TEST_P(TimeSeries, DropFrameNonSealed) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 0);

    ts->AddFrame(0);

    EXPECT_DEATH(
        ts->DropFrame(0),
        "dropping a non-sealed frame");
}

TEST_P(TimeSeries, DropFrameIdxMismatch) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 2, 2);

    EXPECT_DEATH(
        ts->DropFrame(0),
        "frame idx mismatch on drop");

    EXPECT_DEATH(
        ts->DropFrame(2),
        "frame idx mismatch on drop");

    ts->DropFrame(1);
    ts->DropFrame(2); // no frame to drop, but OK
}

TEST_P(TimeSeries, ReadFrame) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto frame = ts->ReadFrame(1);
        EXPECT_EQ(frame.Type(), GetParam());
        EXPECT_EQ(frame.NumPoints(), 0u);

        ExpectSeriesEq(frame, {});
    }

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    {
        auto frame = ts->ReadFrame(1);
        EXPECT_EQ(frame.Type(), GetParam());
        EXPECT_EQ(frame.NumPoints(), 2u);
        ExpectSeriesEq(frame, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});
    }
}

TEST_P(TimeSeries, ReadFrameImmutable) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    auto frame1 = ts->ReadFrame(1);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    auto frame2 = ts->ReadFrame(1);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:20Z", 15)
                    .Done()
                .Done());
        ts->AddPoint(1, data);
    }

    auto frame3 = ts->ReadFrame(1);

    EXPECT_NE(frame1.Data(), frame2.Data());
    EXPECT_NE(frame2.Data(), frame3.Data());
    EXPECT_NE(frame1.Data(), frame3.Data());

    ExpectSeriesEq(frame1, {});
    ExpectSeriesEq(frame2, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});
    ExpectSeriesEq(frame3, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}, {"2000-01-01T00:00:20Z", 15}});
}

TEST_P(TimeSeries, ReadFrameUnsorted) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);

    ExpectSeriesEq(ts->ReadFrame(1), {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    ExpectSeriesEq(ts->ReadFrame(1), {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    {
        auto data = Point(
        NSolomon::NSlog::MakeSlog(10)
            .Generic({{"sensor", "smh"}}, GetParam())
                .Add("2000-01-01T00:00:05Z", 15)
                .Done()
            .Done());
        ts->AddPoint(1, data);
    }

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 1u);

    ExpectSeriesEq(ts->ReadFrame(1), {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:05Z", 15}, {"2000-01-01T00:00:10Z", 10}});

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 2u);

    auto frame = ts->ReadFrame(1);
    ts.Reset();
    ExpectSeriesEq(frame, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:05Z", 15}, {"2000-01-01T00:00:10Z", 10}});
}

TEST_P(TimeSeries, ReadFrameSealed) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 1, 2);

    {
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 5)
                    .Add("2000-01-01T00:00:10Z", 10)
                    .Done()
                .Done());
        ts->AddPoint(1, data);
        ts->SealFrame(1);
    }

    auto frame1 = ts->ReadFrame(1);
    auto frame2 = ts->ReadFrame(1);

    EXPECT_EQ(frame1.Data(), frame2.Data());

    ts.Reset();

    ExpectSeriesEq(frame1, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});
    ExpectSeriesEq(frame2, {{"2000-01-01T00:00:00Z", 5}, {"2000-01-01T00:00:10Z", 10}});

    // EXPECT_EQ(Metrics->StorageMerges->Get(), 0u);
    // EXPECT_EQ(Metrics->StorageReorderings->Get(), 0u);
}

TEST_P(TimeSeries, ReadFrameIdxMismatch) {
    auto ts = TTimeSeries::Create(GetParam(), 1, 2, 2);

    EXPECT_DEATH(
        ts->ReadFrame(0),
        "unknown frame index 0");

    EXPECT_DEATH(
        ts->ReadFrame(2),
        "unknown frame index 2");

    ts->DropFrame(1);

    EXPECT_DEATH(
        ts->ReadFrame(2),
        "unknown frame index 2");
}

TEST_P(TimeSeries, ReadFrameBetween) {
    auto ts = TTimeSeries::Create(GetParam(), 0, 0, 0);

    {
        ts->AddFrame(0);
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:00:00Z", 1)
                    .Add("2000-01-01T00:10:00Z", 2)
                    .Add("2000-01-01T00:20:00Z", 3)
                    .Done()
                .Done());
        ts->AddPoint(0, data);
    }

    {
        ts->AddFrame(1);
    }

    {
        ts->AddFrame(2);
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:10:00Z", 4)
                    .Add("2000-01-01T00:30:00Z", 6)
                    .Done()
                .Done());
        ts->AddPoint(2, data);
    }

    {
        ts->AddFrame(3);
        auto data = Point(
            NSolomon::NSlog::MakeSlog(10)
                .Generic({{"sensor", "smh"}}, GetParam())
                    .Add("2000-01-01T00:40:00Z", 7)
                    .Done()
                .Done());
        ts->AddPoint(3, data);
    }

    auto frames = ts->ReadBetween(
        TInstant::ParseIso8601("2000-01-01T00:10:00Z"),
        TInstant::ParseIso8601("2000-01-01T00:20:01Z"));

    ASSERT_EQ(frames.Frames().size(), 2u);

    ExpectSeriesEq(frames.Frames()[0], {{"2000-01-01T00:00:00Z", 1}, {"2000-01-01T00:10:00Z", 2}, {"2000-01-01T00:20:00Z", 3}});
    ExpectSeriesEq(frames.Frames()[1], {{"2000-01-01T00:10:00Z", 4}, {"2000-01-01T00:30:00Z", 6}});

    ExpectSeriesEq(frames, {{"2000-01-01T00:10:00Z", 4}, {"2000-01-01T00:20:00Z", 3}});
}

INSTANTIATE_TEST_SUITE_P(
    AllTypes,
    TimeSeries,
    testing::Values(
        NSolomon::NTsModel::EPointType::DGauge,
        NSolomon::NTsModel::EPointType::Counter,
        NSolomon::NTsModel::EPointType::Rate,
        NSolomon::NTsModel::EPointType::IGauge,
        NSolomon::NTsModel::EPointType::Hist,
        NSolomon::NTsModel::EPointType::HistRate,
        NSolomon::NTsModel::EPointType::DSummary,
        NSolomon::NTsModel::EPointType::LogHist),
    [](const testing::TestParamInfo<NSolomon::NTsModel::EPointType>& info) -> std::string {
        return ToString(info.param);
    });
