#include "dispatcher.h"

#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#define LOG_P "{dispatcher " << SelfId() << "} "

using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;

namespace NSolomon::NMemStore::NWal {
namespace {

class TDispatcherEvents: private TPrivateEvents {
    enum {
        WriteQueuesClosed = SpaceBegin,
        HostWatchersClosed,
        AllClosed,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TWriteQueuesClosed: public TEventLocal<TWriteQueuesClosed, WriteQueuesClosed> {
    };

    struct THostWatchersClosed: public TEventLocal<THostWatchersClosed, HostWatchersClosed> {
    };

    struct TAllClosed: public TEventLocal<TAllClosed, AllClosed> {
    };
};

class TDispatcher: public TActorBootstrapped<TDispatcher> {
public:
    TDispatcher(
            THashMap<ui64, TTabletData> tablets,
            TVector<TActorId> kvClients,
            TVector<TActorId> watchers,
            TActorId shardManagerId,
            std::shared_ptr<IIndexWriteLimiter> indexLimiter,
            NMonitoring::TMetricRegistry& registry)
        : Tablets_(std::move(tablets))
        , ShardManagerId_(shardManagerId)
        , IndexGlobalLimiter_(std::move(indexLimiter))
        , KvClients_(std::move(kvClients))
        , HostWatchers_(std::move(watchers))
        , Registry_(registry)
    {
        Registry_.IntGauge({{"sensor", "wal.totalHosts"}})->Set(KvClients_.size());
        AliveHostsMetric_ = Registry_.IntGauge({{"sensor", "wal.aliveHosts"}});
        StrayTabletsMetric_ = Registry_.IntGauge({{"sensor", "wal.strayTablets"}});
        AliveHostsMetric_->Set(KvClients_.size());
        StrayTabletsMetric_->Set(TabletList_.size());

        for (auto& [tablet, tabletData]: Tablets_) {
            TabletList_.push_back(tablet);
            LocalTablets_[tabletData.Host].insert(tablet);
        }

        Sort(TabletList_.begin(), TabletList_.end());

        StrayTablets_.insert(TabletList_.begin(), TabletList_.end());
        AliveHosts_ = KvClients_;
    }

public:
    void Bootstrap() {
        Become(&TThis::StateFunc);
        for (auto watcher: HostWatchers_) {
            Send(watcher, new THostWatcherEvents::TSubscribe);
        }
        InitCounter_ = Tablets_.size();
        Y_VERIFY(InitCounter_ > 0, "WAL manager initialization error: no tablets");
        for (auto& [tablet, tabletData]: Tablets_) {
            Send(tabletData.WriteQueue, new TWalEvents::TSetDispatcher(SelfId()));
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::TCommonEvents::TAsyncPoison, Handle)
            hFunc(TWalEvents::TAddLogRecord, Handle)
            hFunc(TWalEvents::TInitDone, OnTabletInitDone)
            hFunc(THostWatcherEvents::THostStatus, Handle)
            hFunc(TWalEvents::TSnapshot, Handle)
            hFunc(TDispatcherEvents::TWriteQueuesClosed, Handle)
            hFunc(TDispatcherEvents::THostWatchersClosed, Handle)
            hFunc(TDispatcherEvents::TAllClosed, Handle)
            hFunc(TWalEvents::TSetWriteHandler, OnSetWriteHandler)
        }
    }

    void Handle(NSolomon::TCommonEvents::TAsyncPoison::TPtr& ev) {
        MON_INFO(Wal, LOG_P << "dispatcher closed, stopping write queues");

        if (Closed_) {
            MON_WARN(Wal, LOG_P << "got multiple AsyncPoison events");
            ev->Get()->Done();
            return;
        }

        Closed_ = true;
        PoisonEvent_ = ev->Release();

        TVector<NThreading::TFuture<void>> futures;

        for (auto& [tablet, tabletData]: Tablets_) {
            auto poison = MakeHolder<NSolomon::TCommonEvents::TAsyncPoison>();
            futures.push_back(poison->Future());
            Send(tabletData.WriteQueue, std::move(poison));
        }

        auto actorSystem = TActorContext::ActorSystem();
        auto selfId = SelfId();

        NThreading::WaitAll(futures).Subscribe([=](auto&&) {
            actorSystem->Send(selfId, new TDispatcherEvents::TWriteQueuesClosed);
        });
    }

    void Handle(TWalEvents::TAddLogRecord::TPtr& ev) {
        if (Closed_) {
            Send(ev->Sender, new TWalEvents::TAddLogRecordResult{
                TApiError{grpc::UNAVAILABLE, "MemStore is shutting down"}}, 0, ev->Cookie);
            return;
        }

        if (InitCounter_ != 0) {
            Send(ev->Sender, new TWalEvents::TAddLogRecordResult{
                    TApiError{grpc::UNAVAILABLE, "MemStore is initializing"}}, 0, ev->Cookie);
            return;
        }

        if (!IndexGlobalLimiter_->CanAddIndexMessage()) {
            Send(ev->Sender, new TWalEvents::TAddLogRecordResult{
                    TApiError{grpc::RESOURCE_EXHAUSTED, "In flight requests count is over the limit"}}, 0, ev->Cookie);
            return;
        }

        auto tablet = TabletList_[ev->Get()->NumId % TabletList_.size()];
        TActivationContext::Send(ev->Forward(Tablets_[tablet].WriteQueue));
    }

    void OnTabletInitDone(TWalEvents::TInitDone::TPtr& ev) {
        Y_VERIFY(InitCounter_ > 0);
        --InitCounter_;
        MON_INFO(Wal, LOG_P "Tablet " << ev->Get()->TabletId << " initialization done. Not initialized tablets counter: " << InitCounter_);
        if (InitCounter_ == 0) {
            TVector<TActorId> writeHandlers;
            writeHandlers.reserve(WriteHandlers_.size());
            for (const auto& actorId: WriteHandlers_) {
                writeHandlers.push_back(actorId);
            }
            Send(ShardManagerId_, new NWal::TWalEvents::TWalInitialized{std::move(writeHandlers)});
            MON_INFO(Wal, LOG_P "All tablets are initialized");
        }
    }

    void Handle(THostWatcherEvents::THostStatus::TPtr& ev) {
        auto host = ev->Get()->HostId;

        bool needShuffle = false;

        if (ev->Get()->IsAlive) {
            auto& localTablets = ev->Get()->LocalTablets;

            if (std::find(AliveHosts_.begin(), AliveHosts_.end(), host) == AliveHosts_.end()) {
                AliveHosts_.push_back(host);
                needShuffle = true;
            }

            for (auto tablet: localTablets) {
                if (Tablets_.contains(tablet)) {
                    StrayTablets_.erase(tablet);
                    MoveTablet(tablet, host);
                }
            }
        } else {
            if (std::find(AliveHosts_.begin(), AliveHosts_.end(), host) != AliveHosts_.end()) {
                AliveHosts_.erase(
                    std::remove(AliveHosts_.begin(), AliveHosts_.end(), host),
                    AliveHosts_.end());
                needShuffle = true;
            }

            auto& localTablets = LocalTablets_[host];
            StrayTablets_.insert(localTablets.begin(), localTablets.end());
        }

        if (needShuffle) {
            ShuffleStrayTablets();
        }

        AliveHostsMetric_->Set(AliveHosts_.size());
        StrayTabletsMetric_->Set(StrayTablets_.size());
    }

    void Handle(TWalEvents::TSnapshot::TPtr& ev) {
        if (Closed_) {
            Send(ev->Get()->Subscriber, new TWalEvents::TSnapshotProcessed);
            return;
        }

        auto tablet = ev->Get()->LogId.TabletId;
        if (!Tablets_.contains(tablet)) {
            MON_WARN(Wal, LOG_P << "snapshot for unknown tablet " << tablet << ", dropping points");
        } else {
            TActivationContext::Send(ev->Forward(Tablets_[tablet].WriteQueue));
        }
    }

    void MoveTablet(ui64 tablet, TActorId to) {
        auto& tabletData = Tablets_[tablet];

        if (tabletData.Host == to) {
            return;
        }

        Send(tabletData.WriteQueue, new THostWatcherEvents::THost{to});

        LocalTablets_[tabletData.Host].erase(tablet);
        LocalTablets_[to].insert(tablet);

        tabletData.Host = to;
    }

    void ShuffleStrayTablets() {
        if (AliveHosts_.empty()) {
            return;
        }

        for (auto tablet: StrayTablets_) {
            auto host = AliveHosts_[RandomNumber(AliveHosts_.size())];
            MoveTablet(tablet, host);
        }
    }

    void Handle(TDispatcherEvents::TWriteQueuesClosed::TPtr&) {
        MON_INFO(Wal, LOG_P << "write queues stopped, stopping host watchers");

        TVector<NThreading::TFuture<void>> futures;

        for (auto hostWatcher: HostWatchers_) {
            auto poison = MakeHolder<NSolomon::TCommonEvents::TAsyncPoison>();
            futures.push_back(poison->Future());
            Send(hostWatcher, std::move(poison));
        }

        auto actorSystem = TActorContext::ActorSystem();
        auto selfId = SelfId();

        NThreading::WaitAll(futures).Subscribe([=](auto&&) {
            actorSystem->Send(selfId, new TDispatcherEvents::THostWatchersClosed);
        });
    }

    void Handle(TDispatcherEvents::THostWatchersClosed::TPtr&) {
        MON_INFO(Wal, LOG_P << "host watchers stopped, stopping KV clients");

        TVector<NThreading::TFuture<void>> futures;

        for (auto client: KvClients_) {
            auto poison = std::make_unique<NSolomon::TCommonEvents::TAsyncPoison>();
            futures.push_back(poison->Future());
            Send(client, poison.release());
        }

        auto actorSystem = TActorContext::ActorSystem();
        auto selfId = SelfId();

        NThreading::WaitAll(futures).Subscribe([=](auto&&) {
            actorSystem->Send(selfId, new TDispatcherEvents::TAllClosed);
        });
    }

    void Handle(TDispatcherEvents::TAllClosed::TPtr&) {
        MON_INFO(Wal, LOG_P << "persister stopped");

        PassAway();
        if (PoisonEvent_) {
            PoisonEvent_->Done();
        }
    }

    void OnSetWriteHandler(NWal::TWalEvents::TSetWriteHandler::TPtr& ev) {
        auto handler = ev->Get()->WriteHandler;
        WriteHandlers_.insert(handler);
        if (InitCounter_ == 0) {
            /**
             * all shards already initialized, so allow write handler to accept requests immediately
             */
            Send(handler, new NWal::TWalEvents::TWalInitialized{});
        }
    }

private:
    TVector<ui64> TabletList_;
    THashMap<ui64, TTabletData> Tablets_;
    THashMap<TActorId, THashSet<ui64>> LocalTablets_;
    i32 InitCounter_{-1};
    TActorId ShardManagerId_;
    std::shared_ptr<IIndexWriteLimiter> IndexGlobalLimiter_;
    // Tablets with unknown host. They are assigned random host.
    THashSet<ui64> StrayTablets_;
    TVector<TActorId> AliveHosts_;
    TVector<TActorId> KvClients_;
    TVector<TActorId> HostWatchers_;
    THashSet<TActorId> WriteHandlers_;
    NMonitoring::TMetricRegistry& Registry_;

    bool Closed_ = false;
    THolder<TCommonEvents::TAsyncPoison> PoisonEvent_ = nullptr;

    NMonitoring::TIntGauge* AliveHostsMetric_;
    NMonitoring::TIntGauge* StrayTabletsMetric_;
};

} // namespace

std::unique_ptr<IActor> CreateDispatcher(
        THashMap<ui64, TTabletData> tablets,
        TVector<TActorId> clients,
        TVector<TActorId> watchers,
        TActorId shardManagerId,
        std::shared_ptr<IIndexWriteLimiter> indexLimiter,
        NMonitoring::TMetricRegistry& registry)
{
    return std::make_unique<TDispatcher>(
            std::move(tablets),
            std::move(clients),
            std::move(watchers),
            shardManagerId,
            std::move(indexLimiter),
            registry);
}

} // namespace NSolomon::NMemStore::NWal
