#include "host_watcher.h"

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/kv/actor_bridge/actor_bridge.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/events.h>

#define LOG_P "{host watcher " << SelfId() << " (" << GetEndpoint() << ")} "

using namespace NActors;

namespace NSolomon::NMemStore::NWal {
namespace {

class THostWatcher: public TActorBootstrapped<THostWatcher> {
public:
    THostWatcher(TActorId client, THostWatcherConfig config)
        : Client_(client)
        , Config_(std::move(config))
    {
    }

public:
    void Bootstrap() {
        Become(&TThis::StateFunc);
        Send(SelfId(), new TEvents::TEvWakeup);
        Send(Client_, new TEvents::TEvSubscribe);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::TCommonEvents::TAsyncPoison, Handle)
            hFunc(TEvents::TEvWakeup, Handle)
            hFunc(NSolomon::NKv::TEvents::TLocalTabletsResponse, Handle)
            hFunc(THostWatcherEvents::TSubscribe, Handle)
            hFunc(THostWatcherEvents::TUnsubscribe, Handle)
        }
    }

    void Handle(NSolomon::TCommonEvents::TAsyncPoison::TPtr& ev) {
        PassAway();
        ev->Get()->Done();
    }

    void Handle(TEvents::TEvWakeup::TPtr&) {
        MON_DEBUG(Wal, LOG_P << "ping");
        Send(Client_, NSolomon::NKv::TEvents::LocalTablets().Release());
    }

    void Handle(NSolomon::NKv::TEvents::TLocalTabletsResponse::TPtr& ev) {
        if (ev->Sender != Client_) {
            MON_WARN(Wal, LOG_P << "received message from unknown client " << ev->Sender);
            return;
        }

        if (ev->Get()->Fail()) {
            MON_DEBUG(Wal, LOG_P << "ping failed, scheduling next ping in " << Config_.RetryInterval);
            Schedule(Config_.RetryInterval, new TEvents::TEvWakeup);

            const TKvClientError& err = ev->Get()->Error();
            if (IsAlive_ && err.IsTransportError() && err.AsTransportError() == grpc::StatusCode::UNAVAILABLE) {
                BecomeUnavailable();
            }
            return;
        }

        if (!IsAlive_) {
            BecomeAvailable();
        }

        MON_DEBUG(Wal, LOG_P << "ping succeed, scheduling next ping in " << Config_.PingInterval);
        Schedule(Config_.PingInterval, new TEvents::TEvWakeup);

        auto localTabletList = ev->Get()->Extract();
        auto localTablets = THashSet<ui64>{localTabletList.begin(), localTabletList.end()};

        if (!Config_.TabletFilter.empty()) {
            size_t numFiltered = 0;
            for (auto it = localTablets.begin(); it != localTablets.end(); ) {
                if (!Config_.TabletFilter.contains(*it)) {
                    localTablets.erase(it++);
                    numFiltered += 1;
                } else {
                    ++it;
                }
            }
            MON_DEBUG(Wal, LOG_P << "filtered " << numFiltered << " local tablets");
        }

        if (localTablets == LocalTablets_) {
            MON_DEBUG(Wal, LOG_P << "local tablets did not change");
            return;
        }

        MON_INFO(Wal, LOG_P << "local tablets changed");
        LocalTablets_ = std::move(localTablets);
        SendUpdate();
    }

    void BecomeUnavailable() {
        MON_INFO(Wal, LOG_P << "host become unavailable");
        IsAlive_ = false;
        LocalTablets_.clear();
        SendUpdate();
    }

    void BecomeAvailable() {
        MON_INFO(Wal, LOG_P << "host become available again");
        IsAlive_ = true;
        SendUpdate();
    }

    void Handle(THostWatcherEvents::TSubscribe::TPtr& ev) {
        Subscribers_.insert(ev->Sender);
        if (StatusIsKnown_) {
            Send(ev->Sender, new THostWatcherEvents::THostStatus{Client_, IsAlive_, LocalTablets_});
        }
    }

    void Handle(THostWatcherEvents::TUnsubscribe::TPtr& ev) {
        Subscribers_.erase(ev->Sender);
    }

    void SendUpdate() {
        for (auto recipient: Subscribers_) {
            Send(recipient, new THostWatcherEvents::THostStatus{Client_, IsAlive_, LocalTablets_});
        }

        StatusIsKnown_ = true;
    }

    TStringBuf GetEndpoint() const {
        return Config_.Endpoint;
    }

private:
    TActorId Client_;
    THostWatcherConfig Config_;
    THashSet<TActorId> Subscribers_;
    bool StatusIsKnown_ = false;
    bool IsAlive_ = true;
    THashSet<ui64> LocalTablets_ = {};
};

} // namespace

std::unique_ptr<IActor> CreateHostWatcher(TActorId client, THostWatcherConfig config) {
    return std::make_unique<THostWatcher>(client, std::move(config));
}

} // namespace NSolomon::NMemStore::NWal
