#pragma once

#include <solomon/services/memstore/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/events/slots.h>
#include <solomon/libs/cpp/actors/fwd.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore::NWal {

class THostWatcherEvents: private TEventSlot<EEventSpace::MemStore, ES_HOST_WATCHER> {
    enum {
        Subscribe = SpaceBegin,
        Unsubscribe,
        HostStatus,
        Host,
        End,
        };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Subscribe on host watcher notifications.
     */
    struct TSubscribe: public NActors::TEventLocal<TSubscribe, Subscribe> {
    };

    /**
     * Unsubscribe from host watcher notifications.
     */
    struct TUnsubscribe: public NActors::TEventLocal<TUnsubscribe, Unsubscribe> {
    };

    /**
     * Notification about current YDB host status.
     */
    struct THostStatus: public NActors::TEventLocal<THostStatus, HostStatus> {
        NActors::TActorId HostId;
        bool IsAlive;
        THashSet<ui64> LocalTablets;

        THostStatus(NActors::TActorId hostId, bool isAlive, THashSet<ui64> localTablets) noexcept
            : HostId{hostId}
            , IsAlive{isAlive}
            , LocalTablets(std::move(localTablets))
        {
        }
    };

    /**
     * Request to switch to a different YDB host.
     */
    struct THost: public NActors::TEventLocal<THost, Host> {
        NActors::TActorId HostId;

        explicit THost(NActors::TActorId hostId) noexcept
            : HostId{hostId}
        {
        }
    };
};

/**
 * Settings for host watcher.
 */
struct THostWatcherConfig {
    /**
     * Refresh list of local tablets once in this amount of time.
     */
    TDuration PingInterval = TDuration::Seconds(10);

    /**
     * If host does not respond, ping it once in this amount of time.
     */
    TDuration RetryInterval = TDuration::Seconds(3);

    /**
     * YDB host endpoint. Used for logging only.
     */
    TString Endpoint = "";

    /**
     * Only track locality of the given tablets. If host contains tablets that are not from our solomon volume,
     * they will be ignored.
     */
    THashSet<ui64> TabletFilter = {};
};

/**
 * Monitors status of a YDB host and sends status updates to subscribers.
 *
 * @param client        id of the KV client used to communicate with host (see `NSolomon::NKv::CreateKvClientActor`).
 * @param config        other settings for host watcher.
 */
std::unique_ptr<NActors::IActor> CreateHostWatcher(NActors::TActorId client, THostWatcherConfig config = {});

} // namespace NSolomon::NMemStore::NWal
