#include "init.h"
#include "dispatcher.h"
#include "write_queue.h"
#include "host_watcher.h"

#include <solomon/services/memstore/lib/config/config.h>

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/kv/actor_bridge/actor_bridge.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actorsystem.h>

using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;

namespace NSolomon::NMemStore::NWal {

TActorId CreateWalManager(
        TActorSystem& actorSystem,
        ui32 executorPool,
        TActorId shardManager,
        const NConfig::TMemStoreConfig& config,
        const std::shared_ptr<IIndexWriteLimiter>& indexLimiter,
        NMonitoring::TMetricRegistry& registry)
{
    ui32 nodeId = NConfig::GetNodeId(config.cluster());
    MON_INFO_C(actorSystem, WalInit, "Initializing WAL manager, nodeId=" << nodeId);

    auto ydbConfig = config.ydb_client();
    if (ydbConfig.GetGrpcConfig().addresses().empty()) {
        ythrow NConfig::TConfigValidationException() << "resolved no YDB hosts";
    } else {
        MON_INFO_C(actorSystem, WalInit, "Working with " << ydbConfig.GetGrpcConfig().addresses_size() << " YDB host(s)");
    }

    auto ydbRpc = NKikimr::CreateClusterGrpc(
            ydbConfig.GetGrpcConfig(),
            registry,
            config.client_id());

    MON_INFO_C(actorSystem, WalInit, "Creating KV clients");
    TVector<TActorId> clients;
    {
        NKv::TRetryOpts clientActorConfig;
//        clientActorConfig.MaxRetries = config.ydb_client().max_retries();
//        clientActorConfig.BackoffTime = FromProtoTime(config.ydb_client().retry_backoff_time());
//        clientActorConfig.BackoffTimeMax = FromProtoTime(config.ydb_client().retry_backoff_time_max());
//        clientActorConfig.BackoffFactor = config.ydb_client().retry_backoff_factor();

        for (auto& address: ydbRpc->Addresses()) {
            auto clientActor = actorSystem.Register(
                NKv::CreateKvClientActor(TString{address}, ydbRpc, clientActorConfig).Release(),
                TMailboxType::HTSwap,
                executorPool);
            clients.emplace_back(clientActor);
        }
    }

    THolder<NKv::TEvents::TResolveTabletsResponse> resolveTabletsResponse;
    for (auto& client: clients) {
        resolveTabletsResponse = actorSystem.Ask<NKv::TEvents::TResolveTabletsResponse>(
            client,
            NKv::TEvents::ResolveTablets(config.volume_path())).ExtractValueSync();
        if (resolveTabletsResponse->Success()) {
            break;
        } else {
            MON_WARN_C(actorSystem, WalInit, "tablets resolution failed: " << resolveTabletsResponse->Error().Message());
        }
    }
    if (resolveTabletsResponse->Fail()) {
        ythrow yexception() << resolveTabletsResponse->Error().Message();
    }
    auto tabletIds = resolveTabletsResponse->Extract();
    Y_ENSURE(!tabletIds.empty(), "no KV tabletIds found");

    MON_INFO_C(actorSystem, WalInit, "Creating write queues");
    THashMap<ui64, TTabletData> tabletData;
    {
        auto globalLimit = FromProtoDataSize(config.write_queue().max_global_queue_size());

        TWriteQueueConfig writeQueueConfig;
        writeQueueConfig.MaxQueueSize = FromProtoDataSize(config.write_queue().max_queue_size());
        writeQueueConfig.Limiter = CreateLimiter(globalLimit);
        writeQueueConfig.BatchSize = FromProtoDataSize(config.write_queue().batch_size());
        writeQueueConfig.WriteTimeout = FromProtoTime(config.write_queue().write_timeout());
        writeQueueConfig.WriteBackoff = FromProtoTime(config.write_queue().write_backoff());
        writeQueueConfig.Index = shardManager;

        registry.IntGauge(
            {{"sensor", "wal.writeQueueGlobalSizeLimit"}})->Set(globalLimit);

        for (auto tabletId: tabletIds) {
            auto client = clients[RandomNumber(clients.size())];
            auto writeQueue = CreateWriteQueue(client, nodeId, tabletId, {}, {}, indexLimiter, registry, writeQueueConfig);
            tabletData.emplace(tabletId, TTabletData {
                actorSystem.Register(writeQueue.release(), TMailboxType::HTSwap, executorPool),
                client
            });
        }
    }

    MON_INFO_C(actorSystem, WalInit, "Creating YDB host watchers");
    TVector<TActorId> watchers;
    {
        THostWatcherConfig hostWatcherConfig;
//        hostWatcherConfig.PingInterval = FromProtoTime(config.ydb_client().host_ping_interval());
//        hostWatcherConfig.RetryInterval = FromProtoTime(config.ydb_client().host_down_ping_interval());
        hostWatcherConfig.TabletFilter = {tabletIds.begin(), tabletIds.end()};

        size_t i = 0;
        for (auto& address: ydbRpc->Addresses()) {
            hostWatcherConfig.Endpoint = address;
            auto watcher = CreateHostWatcher(clients[i], hostWatcherConfig);
            watchers.push_back(actorSystem.Register(watcher.release(), TMailboxType::Simple, executorPool));
        }
    }

    MON_INFO_C(actorSystem, WalInit, "Creating dispatcher");
    auto dispatcher = CreateDispatcher(tabletData, clients, watchers, shardManager, indexLimiter, registry);
    MON_INFO_C(actorSystem, WalInit, "WAL manager initialized");
    return actorSystem.Register(dispatcher.release(), TMailboxType::HTSwap, executorPool);
}

} // namespace NSolomon::NMemStore::NWal
