#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/index/index_limiter.h>

#include <solomon/services/memstore/lib/wal/dispatcher.h>
#include <solomon/services/memstore/lib/wal/host_watcher.h>
#include <solomon/services/memstore/lib/wal/write_queue.h>

#include <solomon/libs/cpp/actors/events/common.h>

#include "fixture.h"

using namespace NSolomon;
using namespace NSolomon::NMemStore;
using namespace NSolomon::NMemStore::NWal;
using namespace NSolomon::NMemStore::NIndex;
using namespace std::string_view_literals;

class DispatcherTest: public TWalFixture<3> {
public:
    void SetUp() override {
        TWalFixture::SetUp();
        IndexLimiter = CreateIndexWriteLimiter(TIndexLimiterConfig::Default());

        THashMap<ui64, TTabletData> tablets;

        Sort(TabletIds);

        for (auto tabletId: TabletIds) {
            Listeners.push_back(Runtime->AllocateEdgeActor());
            Rpcs.push_back(CreateRpc());

            auto clientId = Runtime->Register(NKv::CreateKvClientActor("host-" + ToString(tabletId), Rpcs.back().get(), {}));
            auto writeQueue = Runtime->Register(CreateWriteQueue(clientId, 0, tabletId, 0, 0, IndexLimiter, Registry));

            tablets[tabletId] = {writeQueue, clientId};
            Clients.push_back(clientId);
        }
        NActors::TActorId shardManagerId;
        Dispatcher = Runtime->Register(CreateDispatcher(std::move(tablets), Clients, {}, shardManagerId, IndexLimiter, Registry));
        Runtime->WaitForBootstrap();
    }

    NActors::IEventHandle* CreateAddLogRecordEvent(
        NActors::TActorId writer,
        NActors::TActorId listener,
        TNumId numId,
        TString meta,
        TString data)
    {
        auto* event = new TWalEvents::TAddLogRecord{
            numId,
            TShardKey{"project", "cluster", "service"},
            std::move(meta),
            std::move(data)
        };
        return new NActors::IEventHandle{writer, listener, event};
    }

    void WriteOne(ui64 txn) {
        for (ui32 i = 0; i < TabletIds.size(); ++i) {
            Runtime->Send(CreateAddLogRecordEvent(Dispatcher, Listeners[i], i, "META", "DATA"));
        }

        for (ui32 i = 0; i < TabletIds.size(); ++i) {
            auto ev = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(Listeners[i]);
            ASSERT_TRUE(ev->Get()->Success());
            EXPECT_EQ(ev->Get()->Value(), TLogId(TabletIds[i], txn));
        }
    }

public:
    NActors::TActorId Dispatcher;
    TVector<std::shared_ptr<NKv::TTestRpc>> Rpcs;
    TVector<NActors::TActorId> Clients;
    TVector<NActors::TActorId> Listeners;
    std::shared_ptr<IIndexWriteLimiter> IndexLimiter;
};

TEST_F(DispatcherTest, Dispatch) {
    WriteOne(1);

    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[0], "c.m.txn.last.000").GetValueSync().Value(),
        "\1\0\0\0\0\0\0\0"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[1], "c.m.txn.last.000").GetValueSync().Value(),
        "\1\0\0\0\0\0\0\0"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[2], "c.m.txn.last.000").GetValueSync().Value(),
        "\1\0\0\0\0\0\0\0"sv);
}

TEST_F(DispatcherTest, LocalTablets) {
    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[0], true, {TabletIds[0]}}));
    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[1], true, {TabletIds[1]}}));
    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[2], true, {TabletIds[2]}}));

    {
        WriteOne(1);

        EXPECT_EQ(Rpcs[0]->NumRequests(), 2u);
        EXPECT_EQ(Rpcs[1]->NumRequests(), 2u);
        EXPECT_EQ(Rpcs[2]->NumRequests(), 2u);

        Rpcs[0]->Reset();
        Rpcs[1]->Reset();
        Rpcs[2]->Reset();
    }

    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[0], true, {TabletIds[0], TabletIds[1]}}));

    {
        WriteOne(2);

        EXPECT_EQ(Rpcs[0]->NumRequests(), 4u);
        EXPECT_EQ(Rpcs[1]->NumRequests(), 0u);
        EXPECT_EQ(Rpcs[2]->NumRequests(), 2u);

        Rpcs[0]->Reset();
        Rpcs[1]->Reset();
        Rpcs[2]->Reset();
    }

    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[1], true, {TabletIds[2]}}));

    {
        WriteOne(3);

        EXPECT_EQ(Rpcs[0]->NumRequests(), 4u);
        EXPECT_EQ(Rpcs[1]->NumRequests(), 2u);
        EXPECT_EQ(Rpcs[2]->NumRequests(), 0u);

        Rpcs[0]->Reset();
        Rpcs[1]->Reset();
        Rpcs[2]->Reset();
    }

    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[0], false, {}}));

    {
        WriteOne(4);

        EXPECT_EQ(Rpcs[0]->NumRequests(), 0u);
        EXPECT_EQ(Rpcs[1]->NumRequests() + Rpcs[2]->NumRequests(), 6u);

        Rpcs[0]->Reset();
        Rpcs[1]->Reset();
        Rpcs[2]->Reset();
    }

    Runtime->Send(Dispatcher, {}, THolder(new THostWatcherEvents::THostStatus{Clients[0], true, {TabletIds[1]}}));

    {
        WriteOne(5);

        EXPECT_GE(Rpcs[0]->NumRequests(), 2u);
        EXPECT_GE(Rpcs[1]->NumRequests(), 2u);
        EXPECT_GE(Rpcs[2]->NumRequests(), 0u);

        Rpcs[0]->Reset();
        Rpcs[1]->Reset();
        Rpcs[2]->Reset();
    }
}

TEST_F(DispatcherTest, Poison) {
    Runtime->Send(CreateAddLogRecordEvent(Dispatcher, Listeners[0], 0, "META", "DATA"));
    Runtime->Send(CreateAddLogRecordEvent(Dispatcher, Listeners[1], 1, "META", "DATA"));

    auto poisonEvent = MakeHolder<NSolomon::TCommonEvents::TAsyncPoison>();
    auto poisonFuture = poisonEvent->Future();

    Runtime->Send(Dispatcher, {}, std::move(poisonEvent));

    Runtime->Send(CreateAddLogRecordEvent(Dispatcher, Listeners[2], 3, "META", "DATA"));

    {
        auto ev = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(Listeners[0]);
        ASSERT_TRUE(ev->Get()->Success());
        EXPECT_EQ(ev->Get()->Value(), TLogId(TabletIds[0], 1));
    }

    {
        auto ev = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(Listeners[1]);
        ASSERT_TRUE(ev->Get()->Success());
        EXPECT_EQ(ev->Get()->Value(), TLogId(TabletIds[1], 1));
    }

    {
        auto ev = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(Listeners[2]);
        ASSERT_TRUE(ev->Get()->Fail());
        EXPECT_EQ(ev->Get()->Error().GetMessage(), "MemStore is shutting down");
    }

    EXPECT_TRUE(poisonFuture.HasValue());

    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[0], "c.m.txn.last.000").GetValueSync().Value(),
        "\1\0\0\0\0\0\0\0"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[1], "c.m.txn.last.000").GetValueSync().Value(),
        "\1\0\0\0\0\0\0\0"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletIds[2], "c.m.txn.last.000").GetValueSync().Value(),
        ""sv);
}
