#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/wal/host_watcher.h>

#include "fixture.h"

using namespace NSolomon;
using namespace NSolomon::NMemStore;
using namespace NSolomon::NMemStore::NWal;

class HostWatcherTest: public TWalFixture<3> {
};

TEST_F(HostWatcherTest, TabletsDiscovery) {
    auto tabletsList = StableClient->LocalTablets().GetValueSync().Value();
    auto tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    EXPECT_EQ(tablets.size(), 3u);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();
    auto watcher = Runtime->Register(CreateHostWatcher(client));
    Runtime->WaitForBootstrap();
    Runtime->Send(watcher, listener, MakeHolder<THostWatcherEvents::TSubscribe>());

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }
}

TEST_F(HostWatcherTest, NewTabletsAppear) {
    auto tabletsList = StableClient->LocalTablets().GetValueSync().Value();
    auto tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    EXPECT_EQ(tablets.size(), 3u);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();
    auto watcher = Runtime->Register(CreateHostWatcher(client));
    Runtime->WaitForBootstrap();
    Runtime->Send(watcher, listener, MakeHolder<THostWatcherEvents::TSubscribe>());

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }

    auto path = "/" + Database + "/kv_NewTabletsAppear_2";
    StableClient->CreateSolomonVolume(path, 5).GetValueSync();
    Y_DEFER {
        Rpc->SetHostUp();
        StableClient->DropSolomonVolume(path).GetValueSync();
    };

    {
        tabletsList = StableClient->LocalTablets().GetValueSync().Value();
        EXPECT_EQ(tabletsList.size(), 8u);
        tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    }

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }
}

TEST_F(HostWatcherTest, TabletsFilter) {
    auto tabletsList = StableClient->LocalTablets().GetValueSync().Value();
    auto tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    EXPECT_EQ(tablets.size(), 3u);

    auto path = "/" + Database + "/kv_TabletsFilter_2";
    StableClient->CreateSolomonVolume(path, 5).GetValueSync();
    Y_DEFER {
        Rpc->SetHostUp();
        StableClient->DropSolomonVolume(path).GetValueSync();
    };

    {
        auto newTabletsList = StableClient->LocalTablets().GetValueSync().Value();
        EXPECT_EQ(newTabletsList.size(), 8u);
    }

    THostWatcherConfig config{};
    config.TabletFilter = tablets;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();
    auto watcher = Runtime->Register(CreateHostWatcher(client, config));
    Runtime->WaitForBootstrap();
    Runtime->Send(watcher, listener, MakeHolder<THostWatcherEvents::TSubscribe>());

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }
}

TEST_F(HostWatcherTest, HostFlickers) {
    auto tabletsList = StableClient->LocalTablets().GetValueSync().Value();
    auto tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    EXPECT_EQ(tablets.size(), 3u);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();
    auto watcher = Runtime->Register(CreateHostWatcher(client));
    Runtime->WaitForBootstrap();
    Runtime->Send(watcher, listener, MakeHolder<THostWatcherEvents::TSubscribe>());

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }

    Rpc->SetHostDown();

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(!ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, THashSet<ui64>());
    }

    Rpc->SetHostUp();

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, THashSet<ui64>());
    }
    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }
}

TEST_F(HostWatcherTest, HostFlickersWithNewTablets) {
    auto tabletsList = StableClient->LocalTablets().GetValueSync().Value();
    auto tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    EXPECT_EQ(tablets.size(), 3u);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();
    auto watcher = Runtime->Register(CreateHostWatcher(client));
    Runtime->WaitForBootstrap();
    Runtime->Send(watcher, listener, MakeHolder<THostWatcherEvents::TSubscribe>());

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }

    Rpc->SetHostDown();

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(!ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, THashSet<ui64>());
    }

    Rpc->SetHostUp();

    auto path = "/" + Database + "/kv_HostFlickersWithNewTablets_2";
    StableClient->CreateSolomonVolume(path, 5).GetValueSync();
    Y_DEFER {
        Rpc->SetHostUp();
        StableClient->DropSolomonVolume(path).GetValueSync();
    };

    {
        tabletsList = StableClient->LocalTablets().GetValueSync().Value();
        EXPECT_EQ(tabletsList.size(), 8u);
        tablets = THashSet<ui64>(tabletsList.begin(), tabletsList.end());
    }

    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, THashSet<ui64>());
    }
    {
        auto ev = Runtime->GrabEdgeEvent<THostWatcherEvents::THostStatus>(listener);
        EXPECT_EQ(ev->Get()->HostId, client);
        EXPECT_TRUE(ev->Get()->IsAlive);
        EXPECT_EQ(ev->Get()->LocalTablets, tablets);
    }
}
