#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/wal/write_queue.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/actors/events/common.h>

#include "fixture.h"

using namespace NSolomon;
using namespace NSolomon::NMemStore;
using namespace NSolomon::NMemStore::NWal;
using namespace NSolomon::NMemStore::NIndex;
using namespace std::string_view_literals;

class WriteQueueTest: public TWalFixture<1> {
public:
    void SetUp() override {
        TWalFixture::SetUp();
        IndexLimiter_ = CreateIndexWriteLimiter(TIndexLimiterConfig{
            .MaxTimeSeriesSize = std::numeric_limits<i64>::max(),
            .MaxParsersSize = std::numeric_limits<i64>::max(),
            .MaxIndexMessageCount = std::numeric_limits<i64>::max()});
    }

protected:
    TVector<TString> Ls() {
        TVector<TString> res;
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        for (auto f: files) {
            res.push_back(f.Name);
        }
        return res;
    }

    void CheckSnapshotState(TTxn expectedSnapshotTxn, TTxn expectedLastDeletedWalTxn, const THashMap<TNumId, TTxn>& expectedState) {
        TTxn snapshotTxn, lastDeletedWalTxn;
        THashMap<TNumId, TTxn> state;

        auto val = StableClient->ReadFile(TabletId, "c.m.snapshot.last.013").GetValueSync().Value();
        TStringInput in{val};
        ::LoadMany(&in, snapshotTxn, lastDeletedWalTxn, state);
        EXPECT_EQ(snapshotTxn, expectedSnapshotTxn);
        EXPECT_EQ(lastDeletedWalTxn, expectedLastDeletedWalTxn);
        EXPECT_EQ(state, expectedState);
    }

    NActors::IEventHandle* CreateAddLogRecordEvent(
        NActors::TActorId writer,
        NActors::TActorId listener,
        TNumId numId,
        TString meta,
        TString data)
    {
        auto* event = new TWalEvents::TAddLogRecord{
            numId,
            TShardKey{"project", "cluster", "service"},
            std::move(meta),
            std::move(data)
        };
        return new NActors::IEventHandle{writer, listener, event};
    }

protected:
    std::shared_ptr<IIndexWriteLimiter> IndexLimiter_;
};

TEST_F(WriteQueueTest, TxnDiscovery) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();

    {
        auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, {}, {}, IndexLimiter_, Registry));
        Runtime->WaitForBootstrap();
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
    }

    StableClient->WriteFile(TabletId, "c.m.txn.last.013", TString("\5\0\0\0\0\0\0\0", 8)).GetValueSync();

    {
        auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, {}, {}, IndexLimiter_, Registry));
        Runtime->WaitForBootstrap();
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 6));
    }
}

TEST_F(WriteQueueTest, SimpleWrites) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 1u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\1\0\0\0\0\0\0\0"sv);
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META2", "DATA2"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 2));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, BigWrites) {
    TWriteQueueConfig config;
    config.BatchSize = 27;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000y");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000001.000001z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000y").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DA"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000001z").GetValueSync().Value(),
            "TA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\1\0\0\0\0\0\0\0"sv);
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META2", "DA123456789012345678901234567XXX"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 2));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 5u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000y");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000001.000001z");
        EXPECT_EQ(files[2].Name, "c.m.l.013.00000000000000002.000000y");
        EXPECT_EQ(files[3].Name, "c.m.l.013.00000000000000002.000001y");
        EXPECT_EQ(files[4].Name, "c.m.l.013.00000000000000002.000002z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000y").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\40\0\0\0META2DA"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000001y").GetValueSync().Value(),
            "123456789012345678901234567");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000002z").GetValueSync().Value(),
            "XXX");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, SmallWrites) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 1, "META2", "DATA2"));
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 2, "META3", "DATA3"));
        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        EXPECT_EQ(ev1->Value(), TLogId(TabletId, 1));
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        EXPECT_EQ(ev2->Value(), TLogId(TabletId, 2));
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Success());
        EXPECT_EQ(ev3->Value(), TLogId(TabletId, 2));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\2\0\0\0\1\0\0\0\5\0\0\0\5\0\0\0\2\0\0\0\5\0\0\0\5\0\0\0META2DATA2META3DATA3"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, SmallAndBigWrites) {
    TWriteQueueConfig config;
    config.BatchSize = 30;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 1, "META2", "DATA2"));
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 2, "META3", "DATA3"));
        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        EXPECT_EQ(ev1->Value(), TLogId(TabletId, 1));
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        EXPECT_EQ(ev2->Value(), TLogId(TabletId, 2));
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Success());
        EXPECT_EQ(ev3->Value(), TLogId(TabletId, 2));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000y");
        EXPECT_EQ(files[2].Name, "c.m.l.013.00000000000000002.000001z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000y").GetValueSync().Value(),
            "LI\0\1\2\0\0\0\1\0\0\0\5\0\0\0\5\0\0\0\2\0\0\0\5\0\0\0\5\0"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000001z").GetValueSync().Value(),
            "\0\0META2DATA2META3DATA3"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, QueueOverflow) {
    TWriteQueueConfig config;
    config.MaxQueueSize = 4;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();
    auto errListener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Fail());
        EXPECT_EQ(ev->Error().GetMessage(), "write queue is full");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "MM", "DD"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "", "1"));
        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Fail());
        EXPECT_EQ(ev2->Error().GetMessage(), "write queue is full");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "MM", "DD"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "", "1"));
        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Fail());
        EXPECT_EQ(ev2->Error().GetMessage(), "write queue is full");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "", "1"));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Fail());
        EXPECT_EQ(ev3->Error().GetMessage(), "write queue is full");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "1", "0"));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Fail());
        EXPECT_EQ(ev3->Error().GetMessage(), "write queue is full");
    }
}

TEST_F(WriteQueueTest, GlobalLimiter) {
    auto limiter = CreateLimiter(4);

    TWriteQueueConfig config;
    config.Limiter = limiter;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();
    auto errListener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Fail());
        EXPECT_EQ(ev->Error().GetMessage(), "global queue limit exceeded");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "MM", "DD"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "1", ""));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Fail());
        EXPECT_EQ(ev2->Error().GetMessage(), "global queue limit exceeded");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "MM", "DD"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "", "1"));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Fail());
        EXPECT_EQ(ev2->Error().GetMessage(), "global queue limit exceeded");
    }

    EXPECT_TRUE(limiter->TryAdd(2));

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "M", "DDD"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Fail());
        EXPECT_EQ(ev->Error().GetMessage(), "global queue limit exceeded");
    }

    limiter->Sub(2);

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "1", ""));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Fail());
        EXPECT_EQ(ev3->Error().GetMessage(), "global queue limit exceeded");
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "M", "D"));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "", "1"));

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Fail());
        EXPECT_EQ(ev3->Error().GetMessage(), "global queue limit exceeded");
    }
}

TEST_F(WriteQueueTest, Retries) {
    NKv::TRetryOpts retryOpts;
    retryOpts.MaxRetries = 2;
    retryOpts.BackoffTime = TDuration::MilliSeconds(10);
    retryOpts.BackoffFactor = 1;

    TWriteQueueConfig config;
    config.WriteBackoff = TDuration::MilliSeconds(5);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get(), retryOpts));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));

        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 1u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\1\0\0\0\0\0\0\0"sv);
    }

    Rpc->SetHostDown();

    Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META2", "DATA2"));
    Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META3", "DATA3"));

    Rpc->SetHostUp();

    {
        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        EXPECT_EQ(ev1->Value(), TLogId(TabletId, 2));
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        EXPECT_EQ(ev2->Value(), TLogId(TabletId, 3));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 3u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(files[2].Name, "c.m.l.013.00000000000000003.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000003.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META3DATA3"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\3\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, RetriesDuringFinalization) {
    NKv::TRetryOpts retryOpts;
    retryOpts.MaxRetries = 2;
    retryOpts.BackoffTime = TDuration::MilliSeconds(10);
    retryOpts.BackoffFactor = 1;

    TWriteQueueConfig config;
    config.WriteBackoff = TDuration::MilliSeconds(5);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get(), retryOpts));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto listener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 1u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\1\0\0\0\0\0\0\0"sv);
    }

    Rpc->SetHostDown(1);

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META2", "DATA2"));
        Runtime->DispatchEvents({}, TDuration::Seconds(10));

        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[2].Name, "t.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "t.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\1\0\0\0\0\0\0\0"sv);
    }

    Rpc->SetHostUp();

    {
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 2));
        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, Poison) {
    NKv::TRetryOpts retryOpts;
    retryOpts.MaxRetries = 2;
    retryOpts.BackoffTime = TDuration::MilliSeconds(10);
    retryOpts.BackoffFactor = 1;

    TWriteQueueConfig config;
    config.WriteBackoff = TDuration::MilliSeconds(5);

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get(), retryOpts));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();

    auto okListener = Runtime->AllocateEdgeActor();
    auto errListener = Runtime->AllocateEdgeActor();
    auto poisonEvent = MakeHolder<NSolomon::TCommonEvents::TAsyncPoison>();
    auto poisonFuture = poisonEvent->Future();

    Rpc->SetHostDown(0, 20);

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "META1", "DATA1"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 0, "META2", "DATA2"));
        Runtime->Send(writer, {}, std::move(poisonEvent));
        Runtime->Send(CreateAddLogRecordEvent(writer, errListener, 0, "META3", "DATA3"));

        Runtime->DispatchEvents();

        auto h1 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev1 = h1->Get();
        ASSERT_TRUE(ev1->Success());
        auto h2 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev2 = h2->Get();
        ASSERT_TRUE(ev2->Success());
        auto h3 = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(errListener);
        auto ev3 = h3->Get();
        ASSERT_TRUE(ev3->Fail());
        EXPECT_EQ(ev3->Error().GetMessage(), "write queue is closed");

        EXPECT_TRUE(poisonFuture.HasValue());

        auto files = StableClient->ListFiles(TabletId).GetValueSync().Value();
        ASSERT_EQ(files.size(), 2u + 1u);
        EXPECT_EQ(files[0].Name, "c.m.l.013.00000000000000001.000000z");
        EXPECT_EQ(files[1].Name, "c.m.l.013.00000000000000002.000000z");
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000001.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.l.013.00000000000000002.000000z").GetValueSync().Value(),
            "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
        EXPECT_EQ(
            StableClient->ReadFile(TabletId, "c.m.txn.last.013").GetValueSync().Value(),
            "\2\0\0\0\0\0\0\0"sv);
    }
}

TEST_F(WriteQueueTest, SnapshotsSimple) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, {}));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 5, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value().Txn, 1u);
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 5, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value().Txn, 2u);
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 5, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value().Txn, 3u);
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 5, "META", "DATA"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value().Txn, 4u);
    }

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000001.000000z",
        "c.m.l.013.00000000000000002.000000z",
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.txn.last.013"
    }));

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(5, TLogId{13, 1}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000002.000000z",
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(1, 1, {{5, 1}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(5, TLogId{13, 3}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000004.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(2, 3, {{5, 3}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(5, TLogId{13, 4}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.s.013.00000000000000003.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(3, 4, {});
}

TEST_F(WriteQueueTest, SnapshotsComplex) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, {}));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "META", "DATA"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "META", "DATA"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 2, "META", "DATA"));
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 1u);
        }
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 2u);
        }
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 2u);
        }
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "META", "DATA"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "META", "DATA"));
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 2, "META", "DATA"));
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 3u);
        }
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 4u);
        }
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 4u);
        }
    }

    {
        Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "META", "DATA"));
        {
            auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
            auto ev = h->Get();
            ASSERT_TRUE(ev->Success());
            EXPECT_EQ(ev->Value().Txn, 5u);
        }
    }

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000001.000000z",
        "c.m.l.013.00000000000000002.000000z",
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.l.013.00000000000000005.000000z",
        "c.m.txn.last.013"
    }));

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 2}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000002.000000z",
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.l.013.00000000000000005.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(1, 1, {{1, 2}, {2, 1}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(2, TLogId{13, 2}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.l.013.00000000000000005.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(2, 2, {{1, 2}, {2, 2}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 4}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000003.000000z",
        "c.m.l.013.00000000000000004.000000z",
        "c.m.l.013.00000000000000005.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.s.013.00000000000000003.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(3, 2, {{1, 4}, {2, 2}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(2, TLogId{13, 4}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000005.000000z",
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.s.013.00000000000000003.000000z",
        "c.m.s.013.00000000000000004.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(4, 4, {{1, 4}});

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 5}, "META", "DATA", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.s.013.00000000000000003.000000z",
        "c.m.s.013.00000000000000004.000000z",
        "c.m.s.013.00000000000000005.000000z",
        "c.m.snapshot.last.013",
        "c.m.txn.last.013"
    }));
    CheckSnapshotState(5, 5, {});
}

TEST_F(WriteQueueTest, SnapshotsWrites) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, {}));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(0, TLogId{13, 1}, "META1", "DATA1", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000z",
        "c.m.snapshot.last.013"
    }));
    CheckSnapshotState(1, 0, {});
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000001.000000z").GetValueSync().Value(),
        "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DATA1"sv);

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(0, TLogId{13, 1}, "META2", "DATA2", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000z",
        "c.m.s.013.00000000000000002.000000z",
        "c.m.snapshot.last.013"
    }));
    CheckSnapshotState(2, 0, {});
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000002.000000z").GetValueSync().Value(),
        "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DATA2"sv);
}

TEST_F(WriteQueueTest, SnapshotsBigWrites) {
    TWriteQueueConfig config;
    config.BatchSize = 27;

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, config));
    Runtime->WaitForBootstrap();
    auto okListener = Runtime->AllocateEdgeActor();

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(0, TLogId{13, 1}, "META1", "DATA1", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000y",
        "c.m.s.013.00000000000000001.000001z",
        "c.m.snapshot.last.013"
    }));
    CheckSnapshotState(1, 0, {});
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000001.000000y").GetValueSync().Value(),
        "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META1DA"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000001.000001z").GetValueSync().Value(),
        "TA1"sv);

    Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(0, TLogId{13, 1}, "META2", "DATA2", okListener));
    Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.s.013.00000000000000001.000000y",
        "c.m.s.013.00000000000000001.000001z",
        "c.m.s.013.00000000000000002.000000y",
        "c.m.s.013.00000000000000002.000001z",
        "c.m.snapshot.last.013"
    }));
    CheckSnapshotState(2, 0, {});
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000002.000000y").GetValueSync().Value(),
        "LI\0\1\1\0\0\0\0\0\0\0\5\0\0\0\5\0\0\0META2DA"sv);
    EXPECT_EQ(
        StableClient->ReadFile(TabletId, "c.m.s.013.00000000000000002.000001z").GetValueSync().Value(),
        "TA2"sv);
}

TEST_F(WriteQueueTest, SnapshotsRestore) {
    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto okListener = Runtime->AllocateEdgeActor();

    {
        auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, 0, 0, IndexLimiter_, Registry, {}));
        Runtime->WaitForBootstrap();

        {
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "MM_00", "DD_00"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "MM_01", "DD_01"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 2, "MM_02", "DD_02"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 3, "MM_03", "DD_03"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 3, "MM_04", "DD_04"));
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 1u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 2u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 2u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 2u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 2u);
            }
        }

        {
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "MM_05", "DD_05"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 1, "MM_06", "DD_06"));
            Runtime->Send(CreateAddLogRecordEvent(writer, okListener, 2, "MM_07", "DD_07"));
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 3u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 4u);
            }
            {
                auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(okListener);
                auto ev = h->Get();
                ASSERT_TRUE(ev->Success());
                EXPECT_EQ(ev->Value().Txn, 4u);
            }
        }

        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(2, TLogId{13, 2}, "11111", "22222", okListener));
        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 1}, "33333", "44444", okListener));
        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 2}, "33333", "44444", okListener));
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

        Runtime->Send(writer, MakeHolder<TCommonEvents::TAsyncPoison>());
    }

    {
        TWriteQueueConfig config;
        config.Index = Runtime->AllocateEdgeActor();
        auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, {}, {}, IndexLimiter_, Registry, config));
        Runtime->WaitForBootstrap();

        {
            auto ev = Runtime->GrabEdgeEvent<NIndex::TShardManagerEvents::TIndexBatch>(config.Index);
            EXPECT_EQ(ev->Get()->LogId, (TLogId{TabletId, 2}));

            auto& perShardRequests = ev->Get()->Requests;
            ASSERT_EQ(perShardRequests.size(), 1u);
            ASSERT_TRUE(perShardRequests.contains(3));

            auto& requests = perShardRequests[3].Requests;
            ASSERT_EQ(requests.size(), 2u);
            EXPECT_EQ(requests[0].Meta, "MM_03");
            EXPECT_EQ(requests[0].Data, "DD_03");
            EXPECT_EQ(requests[1].Meta, "MM_04");
            EXPECT_EQ(requests[1].Data, "DD_04");
        }
        {
            auto ev = Runtime->GrabEdgeEvent<NIndex::TShardManagerEvents::TIndexBatch>(config.Index);
            EXPECT_EQ(ev->Get()->LogId, (TLogId{TabletId, 3}));

            auto& perShardRequests = ev->Get()->Requests;
            ASSERT_EQ(perShardRequests.size(), 1u);
            ASSERT_TRUE(perShardRequests.contains(1));

            auto& requests = perShardRequests[1].Requests;
            ASSERT_EQ(requests.size(), 1u);
            EXPECT_EQ(requests[0].Meta, "MM_05");
            EXPECT_EQ(requests[0].Data, "DD_05");
        }
        {
            auto ev = Runtime->GrabEdgeEvent<NIndex::TShardManagerEvents::TIndexBatch>(config.Index);
            EXPECT_EQ(ev->Get()->LogId, (TLogId{TabletId, 4}));

            auto& perShardRequests = ev->Get()->Requests;
            ASSERT_EQ(perShardRequests.size(), 2u);
            {
                ASSERT_TRUE(perShardRequests.contains(1));
                auto& requests = perShardRequests[1].Requests;
                ASSERT_EQ(requests.size(), 1u);
                EXPECT_EQ(requests[0].Meta, "MM_06");
                EXPECT_EQ(requests[0].Data, "DD_06");
            }
            {
                ASSERT_TRUE(perShardRequests.contains(2));
                auto& requests = perShardRequests[2].Requests;
                ASSERT_EQ(requests.size(), 1u);
                EXPECT_EQ(requests[0].Meta, "MM_07");
                EXPECT_EQ(requests[0].Data, "DD_07");
            }
        }

        EXPECT_EQ(Ls(), TVector<TString>({
            "c.m.l.013.00000000000000002.000000z",
            "c.m.l.013.00000000000000003.000000z",
            "c.m.l.013.00000000000000004.000000z",
            "c.m.s.013.00000000000000001.000000z",
            "c.m.s.013.00000000000000002.000000z",
            "c.m.snapshot.last.013",
            "c.m.txn.last.013"
        }));
        CheckSnapshotState(2, 1, {{1, 2}, {2, 2}, {3, 1}});

        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(1, TLogId{13, 4}, "META", "DATA", okListener));
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

        EXPECT_EQ(Ls(), TVector<TString>({
            "c.m.l.013.00000000000000002.000000z",
            "c.m.l.013.00000000000000003.000000z",
            "c.m.l.013.00000000000000004.000000z",
            "c.m.s.013.00000000000000001.000000z",
            "c.m.s.013.00000000000000002.000000z",
            "c.m.s.013.00000000000000003.000000z",
            "c.m.snapshot.last.013",
            "c.m.txn.last.013"
        }));
        CheckSnapshotState(3, 1, {{1, 4}, {2, 2}, {3, 1}});

        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(3, TLogId{13, 2}, "META", "DATA", okListener));
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

        EXPECT_EQ(Ls(), TVector<TString>({
            "c.m.l.013.00000000000000003.000000z",
            "c.m.l.013.00000000000000004.000000z",
            "c.m.s.013.00000000000000001.000000z",
            "c.m.s.013.00000000000000002.000000z",
            "c.m.s.013.00000000000000003.000000z",
            "c.m.s.013.00000000000000004.000000z",
            "c.m.snapshot.last.013",
            "c.m.txn.last.013"
        }));
        CheckSnapshotState(4, 2, {{1, 4}, {2, 2}});

        Runtime->Send(writer, okListener, MakeHolder<TWalEvents::TSnapshot>(2, TLogId{13, 4}, "META", "DATA", okListener));
        Runtime->GrabEdgeEvent<TWalEvents::TSnapshotProcessed>(okListener);

        EXPECT_EQ(Ls(), TVector<TString>({
            "c.m.s.013.00000000000000001.000000z",
            "c.m.s.013.00000000000000002.000000z",
            "c.m.s.013.00000000000000003.000000z",
            "c.m.s.013.00000000000000004.000000z",
            "c.m.s.013.00000000000000005.000000z",
            "c.m.snapshot.last.013",
            "c.m.txn.last.013"
        }));
        CheckSnapshotState(5, 4, {});
    }
}

TEST_F(WriteQueueTest, SnapshotsDropTmpFiles) {
    StableClient->WriteFile(TabletId, "t.m.l.013.00000000000000001.000000z", TString("contents")).GetValueSync();
    StableClient->WriteFile(TabletId, "t.m.s.013.00000000000000001.000000z", TString("contents")).GetValueSync();
    StableClient->WriteFile(TabletId, "t.m.l.012.00000000000000001.000000z", TString("contents")).GetValueSync();
    StableClient->WriteFile(TabletId, "t.m.s.012.00000000000000001.000000z", TString("contents")).GetValueSync();

    auto client = Runtime->Register(NKv::CreateKvClientActor("host-a", Rpc.get()));
    auto listener = Runtime->AllocateEdgeActor();

    {
        auto writer = Runtime->Register(CreateWriteQueue(client, 13, TabletId, {}, {}, IndexLimiter_, Registry));
        Runtime->WaitForBootstrap();
        Runtime->Send(CreateAddLogRecordEvent(writer, listener, 0, "META1", "DATA1"));
        auto h = Runtime->GrabEdgeEvent<TWalEvents::TAddLogRecordResult>(listener);
        auto ev = h->Get();
        ASSERT_TRUE(ev->Success());
        EXPECT_EQ(ev->Value(), TLogId(TabletId, 1));
    }

    EXPECT_EQ(Ls(), TVector<TString>({
        "c.m.l.013.00000000000000001.000000z",
        "c.m.txn.last.013",
        "t.m.l.012.00000000000000001.000000z",
        "t.m.s.012.00000000000000001.000000z"
    }));
}
