#pragma once

#include <solomon/services/memstore/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/shard_key/shard_key.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore::NWal {

class TWalEvents: private TEventSlot<EEventSpace::MemStore, ES_WAL> {
    enum {
        Snapshot = SpaceBegin,
        SnapshotProcessed,
        AddLogRecord,
        AddLogRecordResult,
        SetDispatcher,
        SetWriteHandler,
        WalInitialized,
        InitDone,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * A new snapshot is ready.
     */
    struct TSnapshot: public NActors::TEventLocal<TSnapshot, Snapshot> {
        /**
         * Num id of the shard from which this snapshot comes from.
         */
        TNumId NumId;

        /**
         * Latest log id stored in the snapshot.
         */
        TLogId LogId;

        /**
         * Meta in slog format.
         */
        TString Meta;

        /**
         * Data in slog format.
         */
        TString Data;

        /**
         * See `TSnapshotProcessed`.
         */
        NActors::TActorId Subscriber;

        TSnapshot(TNumId numId, TLogId logId, TString meta, TString data, NActors::TActorId subscriber = {})
            : NumId(numId)
            , LogId(logId)
            , Meta(std::move(meta))
            , Data(std::move(data))
            , Subscriber(subscriber)
        {
        }
    };

    /**
     * Wal manager can send a notification when snapshot is persisted.
     * Index system is not interested in these notifications, but you can use them for testing purposes.
     */
    struct TSnapshotProcessed: public NActors::TEventLocal<TSnapshotProcessed, SnapshotProcessed> {
    };

    /**
     * Request to append a new record to the write-ahead log.
     */
    struct TAddLogRecord: public NActors::TEventLocal<TAddLogRecord, AddLogRecord> {
        TNumId NumId;
        TShardKey ShardKey;
        TString MetaData;
        TString Data;

        TAddLogRecord(TNumId numId, TShardKey shardKey,  TString metaData, TString data)
            : NumId{numId}
            , ShardKey{std::move(shardKey)}
            , MetaData(std::move(metaData))
            , Data(std::move(data))
        {
        }
    };

    /**
     * Result of record append request.
     *
     * Contains an identifier of the record or error message if record append failed.
     * Note that WAL writes may be batched, thus multiple records may have the same id.
     */
    struct TAddLogRecordResult:
            public NActors::TEventLocal<TAddLogRecordResult, AddLogRecordResult>,
            public TErrorOr<TLogId, TApiError>
    {
        explicit TAddLogRecordResult(TErrorOr<TLogId, TApiError> result)
            : TErrorOr(std::move(result))
        {
        }
    };

    /**
     * Report the Id of WAL manager to the write queue instance.
     */
    struct TSetDispatcher: public NActors::TEventLocal<TSetDispatcher, SetDispatcher> {
        NActors::TActorId Dispatcher;

        explicit TSetDispatcher(NActors::TActorId dispatcher)
            : Dispatcher(dispatcher)
        {
        }
    };

    /**
     * Report the Id of GRPC write handler to the WAL manager
     */

    struct TSetWriteHandler: public NActors::TEventLocal<TSetWriteHandler, SetWriteHandler> {
        NActors::TActorId WriteHandler;

        explicit TSetWriteHandler(NActors::TActorId writeHandler)
            : WriteHandler(writeHandler)
        {
        }
    };

    /**
     * Report that WAL is initialized to GRPC write handler
     */
    struct TWalInitialized: public NActors::TEventLocal<TWalInitialized, WalInitialized> {
        // optional
        TVector<NActors::TActorId> ForwardTo;

        explicit TWalInitialized(TVector<NActors::TActorId> forwardTo = {})
            : ForwardTo(std::move(forwardTo))
        {
        }
    };

    /**
     * Write queue reports to WAL manager that all logs are read after memstore start,
     * and metrics data contained in these logs are sent to index.
     */
    struct TInitDone: public NActors::TEventLocal<TInitDone, InitDone> {
        TTabletId TabletId;

        explicit TInitDone(TTabletId tabletId) : TabletId(tabletId) {};
    };
};

} // namespace NSolomon::NMemStore::NWal
