#include "write_queue.h"

#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>
#include <solomon/services/memstore/lib/storage/file_name.h>
#include <solomon/services/memstore/lib/wal/host_watcher.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/kv/actor_bridge/actor_bridge.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/slog/slog.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/util/rope.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/ysaveload.h>
#include <util/generic/deque.h>
#include <util/string/printf.h>

using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;

namespace NSolomon::NMemStore::NWal {
namespace {

constexpr ui64 MAX_INDEX_INFLIGHT = 32ull * 1024 * 1024;
constexpr TDuration MAX_CLEAR_REQUEST_DURATION = TDuration::Seconds(60);
const TDuration MIN_CLEAR_WAL_FILES_PERIOD = MAX_CLEAR_REQUEST_DURATION * 2;
constexpr TDuration INIT_READ_WAL_FILE_PERIOD = TDuration::Seconds(15);

class TWriteQueue: public TActorBootstrapped<TWriteQueue>, private TPrivateEvents {
#define LOG_P "{writer queue " << SelfId() << " (" << TabletId_ << ")} "

private:
    // State machine

    // We write wal and snapshots in a similar manner:
    //
    // - first, we load all queued write requests into a stage and split them into files, 20mb each;
    // - then, we write each file using a temp prefix. We write them sequentially, i.e. not in parallel;
    // - finally, we rename files from temp prefix to the normal prefix. In the same request we update the latest
    //   known txn. This way we ensure atomicity of write.
    enum EState {
        StateNotInited,
        StateIdle,
        StateWrite,
        StateWriteInProgress,
        StateWriteDone,
        StateFinalize,
        StateFinalizeInProgress,
        StateFinalizeDone,
    };

private:
    // Events

    enum EEvent {
        EvRetryInit = SpaceBegin,
        EvRetryWalRead,
        EvRetryWalWrite,
        EvRetryWalFinalize,
        EvRetrySnapshotWrite,
        EvRetrySnapshotFinalize,
        EvClearWalFiles,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");
    struct TEvRetryInit: public TEventLocal<TEvRetryInit, EvRetryInit> {};
    struct TEvRetryWalRead: public TEventLocal<TEvRetryWalRead, EvRetryWalRead> {};
    struct TEvRetryWalWrite: public TEventLocal<TEvRetryWalWrite, EvRetryWalWrite> {};
    struct TEvRetryWalFinalize: public TEventLocal<TEvRetryWalFinalize, EvRetryWalFinalize> {};
    struct TEvRetrySnapshotWrite: public TEventLocal<TEvRetrySnapshotWrite, EvRetrySnapshotWrite> {};
    struct TEvRetrySnapshotFinalize: public TEventLocal<TEvRetrySnapshotFinalize, EvRetrySnapshotFinalize> {};
    struct TEvClearWalFiles: public TEventLocal<TEvClearWalFiles, EvClearWalFiles> {};

    enum EKvCookie {
        KvCookieStateRead,
        KvCookieWalRead,
        KvCookieWalWrite,
        KvCookieWalFinalize,
        KvCookieSnapshotWrite,
        KvCookieSnapshotFinalize,
        KvCookieClearWalFiles
    };

private:
    // Settings

    const ui32 NodeId_;
    const TTabletId TabletId_;
    const TWriteQueueConfig Config_;
    const TString CurrentWalTxnFile_;
    const TString CurrentSnapshotTxnFile_;
    TActorId Client_;

public:
    TWriteQueue(
            TActorId client,
            ui32 nodeId,
            ui64 tabletId,
            TMaybe<ui64> txn,
            TMaybe<ui64> snapshotTxn,
            TWriteQueueConfig config,
            std::shared_ptr<IIndexWriteLimiter> indexWriteLimiter,
            NMonitoring::TMetricRegistry& registry)
        : NodeId_{nodeId}
        , TabletId_{tabletId}
        , Config_{std::move(config)}
        , CurrentWalTxnFile_(WalTxnFilename(NodeId_))
        , CurrentSnapshotTxnFile_(SnapshotTxnFilename(NodeId_))
        , Client_{client}
        , IndexLimiter_(registry)
        , IndexGlobalLimiter_(std::move(indexWriteLimiter))
    {
        VerifyConfig();

        if (txn.Defined()) {
            WalTxn_ = *txn;
            WalState_ = StateIdle;
        }
        if (snapshotTxn.Defined()) {
            SnapshotTxn_ = *txn;
            SnapshotState_ = StateIdle;
        }
        GlobalQueueSizeMetric_ = registry.IntGauge({{"sensor", "wal.writeQueueGlobalSize"}});
        QueueSizeMetric_ = registry.IntGauge(
                {{"sensor", "wal.writeQueueSize"}, {"host", "const"}, {"tabletId", ToString(TabletId_)}});
        GlobalSnapshotQueueSizeMetric_ = registry.IntGauge({{"sensor", "wal.snapshotQueueGlobalSize"}});
        SnapshotQueueSizeMetric_ = registry.IntGauge(
                {{"sensor", "wal.snapshotQueueSize"}, {"host", "const"}, {"tabletId", ToString(TabletId_)}});

        ShapshotStartRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.startsPerSecond"}});
        SnapshotOnIdleRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.onIdlePerSecond"}});
        SnapshotOnWriteRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.onWritePerSecond"}});
        SnapshotWriteDoneRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.writeDonePerSecond"}});
        SnapshotOnWriteDoneRpsMetric_ =  registry.Rate({{"sensor", "wal.snapshot.onWriteDonePerSecond"}});
        SnapshotOnFinalizeRpsMetric_ =  registry.Rate({{"sensor", "wal.snapshot.onFinalizePerSecond"}});
        ShapshotFinalizeDoneRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.finalizeDonePerSecond"}});
        ShapshotOnFinalizeDoneRpsMetric_ = registry.Rate({{"sensor", "wal.snapshot.onFinalizeDonePerSecond"}});

        WriteRpsMetric_ = registry.Rate({{"sensor", "wal.writesPerSecond"}});
        WriteErrRpsMetric_ = registry.Rate({{"sensor", "wal.writeErrorsPerSecond"}});
        WriteSizeRpsMetric_ = registry.Rate({{"sensor", "wal.writeSizePerSecond"}});
        RejectedEvents_ = registry.Rate({{"sensor", "wal.rejectedEvents"}});
        EventProcessingTime_ = registry.HistogramRate(
                {{"sensor", {"wal.eventProcessingTimeMs"}}}, NMonitoring::ExponentialHistogram(16, 2, 1));

        WalTxnMetric_ = registry.IntGauge({{"sensor", "wal.txn.tablet.current"}, {"tabletId", ToString(TabletId_)}});
        LastDeletedWalTxnMetric_ = registry.IntGauge({{"sensor", "wal.txn.tablet.lastDeleted"}, {"tabletId", ToString(TabletId_)}});
        UnsnapshotTxnTabletMetric_ = registry.IntGauge({{"sensor", "wal.txn.tablet.unsnapshotCount"}, {"tabletId", ToString(TabletId_)}});
        UnsnapshotTxnCountMetric_ = registry.IntGauge({{"sensor", "wal.txn.unsnapshotCount"}});

        ResetWalTxnMetric_ = registry.IntGauge({{"sensor", "wal.txn.reset_count"}});
        ResetLastDeletedWalTxnMetric1_ = registry.IntGauge({{"sensor", "wal.txn.resetLastDeletedCount1"}});
        ResetLastDeletedWalTxnMetric2_ = registry.IntGauge({{"sensor", "wal.txn.resetLastDeletedCount2"}});
        ResetLastDeletedWalTxnMetric3_ = registry.IntGauge({{"sensor", "wal.txn.resetLastDeletedCount3"}});

        ClearWalCountMetric_ = registry.IntGauge({{"sensor", "wal.txn.clearWalCount"}});
        WalMemorySizeBytesMetric_ = registry.IntGauge({{"sensor", "wal.inMemorySizeBytes"}});
        ClearErrRpsMetric_ = registry.Rate({{"sensor", "wal.txn.clearErrRate"}});
    }

    void Bootstrap() {
        Become(&TWriteQueue::Main);
        if (WalState_ == StateNotInited || SnapshotState_ == StateNotInited) {
            Init1LoadState();
        }
    }

private:
    void VerifyConfig() {
        Y_VERIFY(
                Config_.ClearWalFilesPeriod >= MIN_CLEAR_WAL_FILES_PERIOD,
                "Clear WAL file period must be greater than %ul seconds",
                (ui32)MIN_CLEAR_WAL_FILES_PERIOD.Seconds() );
    }

    STATEFN(Main) {
        if (NKv::TEvents::IsKvEvent(ev->GetTypeRewrite())) {
            switch (ev->Cookie) {
                case KvCookieStateRead:
                    Init1LoadStateDone(*ev);
                    return;
                case KvCookieWalRead:
                    Init2ReadWalFileDone(*ev);
                    return;
                case KvCookieWalWrite:
                    WalWriteDone(*ev);
                    return;
                case KvCookieWalFinalize:
                    WalFinalizeDone(*ev);
                    return;
                case KvCookieSnapshotWrite:
                    SnapshotWriteDone(*ev);
                    return;
                case KvCookieSnapshotFinalize:
                    SnapshotFinalizeDone(*ev);
                    return;
                case KvCookieClearWalFiles:
                    ClearWalFilesDone(*ev);
                    return;
            }
        }

        switch (ev->GetTypeRewrite()) {
            hFunc(TCommonEvents::TAsyncPoison, OnAsyncPoison);
            hFunc(THostWatcherEvents::THost, OnHost);
            hFunc(NIndex::TShardEvents::TIndexDone, OnIndexDone);
            sFunc(TEvRetryInit, Init1LoadState);
            sFunc(TEvRetryWalRead, Init2ReadWalFile);
            hFunc(TWalEvents::TSetDispatcher, OnSetDispatcher);
            hFunc(TWalEvents::TAddLogRecord, OnAddLogRecord);
            sFunc(TEvRetryWalWrite, WalWriteRetry);
            sFunc(TEvRetryWalFinalize, WalFinalizeRetry);
            hFunc(TWalEvents::TSnapshot, OnSnapshot);
            sFunc(TEvRetrySnapshotWrite, SnapshotWriteRetry);
            sFunc(TEvRetrySnapshotFinalize, SnapshotFinalizeRetry);
            sFunc(TEvClearWalFiles, ClearWalFiles);
        }
    }

private:
    void OnHost(THostWatcherEvents::THost::TPtr& ev) {
        Client_ = ev->Get()->HostId;
        MON_DEBUG(Wal, LOG_P << "set new host: " << Client_);
    }

private:
    // Tear down routine:

    bool Closed_ = false;
    THolder<TCommonEvents::TAsyncPoison> PoisonEvent_ = nullptr;

    void OnAsyncPoison(TCommonEvents::TAsyncPoison::TPtr& ev) {
        MON_DEBUG(Wal, LOG_P << "write queue closed");
        Closed_ = true;
        PoisonEvent_ = ev->Release();

        MaybePassAway();
    }

    void MaybePassAway() {
        if (Closed_ &&
            WalState_ == StateIdle &&
            WalQueue_.empty() &&
            SnapshotState_ == StateIdle &&
            SnapshotQueue_.empty())
        {
            PassAway();
            if (PoisonEvent_) {
                PoisonEvent_->Done();
            }
            ReportUnsnapshotCount(true);
        }
    }

private:
    // Initialization routine:
    //
    // 1: load txn and shard snapshotting state from the tablet;
    // 2: read all available wal records;
    // 3: init done, start writing wal and snapshots as usual.

    TTxn InitCurrentWalFile_ = 0;
    ui32 InitCurrentWalFileIndex_ = 0;
    TString InitCurrentWalFileContents_;
    bool InitIsReadWal_{false};

    void Init1LoadState() {
        MON_DEBUG(Wal, LOG_P << "loading wal state");

        TKikimrKvBatchRequest batch{TabletId_};
        batch.ReadFile(CurrentWalTxnFile_);
        auto walPrefix = WalFilenamePrefixShort(NodeId_, /* tmp = */ true);
        batch.RemoveFiles({walPrefix, walPrefix + "~"});
        batch.ReadFile(CurrentSnapshotTxnFile_);
        auto snapshotPrefix = SnapshotFilenamePrefixShort(NodeId_, /* tmp = */ true);
        batch.RemoveFiles({snapshotPrefix, snapshotPrefix + "~"});
        Send(Client_, NKv::TEvents::Batch(std::move(batch)).Release(), 0, /* cookie = */ KvCookieStateRead);
    }

    void Init1LoadStateDone(IEventHandle& handle) {
        auto ev = handle.Get<NKv::TEvents::TBatchResponse>();

        if (ev->Fail()) {
            MON_ERROR(Wal, LOG_P << "loading wal state error: " << ev->Error().Message());
            Schedule(Config_.FetchTxnBackoff, new TEvRetryInit{});
            return;
        } else {
            MON_DEBUG(Wal, LOG_P << "loading wal state done");

            // Load wal txn
            {
                auto serialized = ev->Value().ReadFileQueryResults.at(0);
                if (!serialized.empty()) {
                    TStringInput in{serialized};
                    ::Load(&in, WalTxn_);
                } else {
                    ResetWalTxnMetric_->Inc();
                }
            }
            WalTxnMetric_->Set(WalTxn_);

            // Load snapshot txn
            {
                auto serialized = ev->Value().ReadFileQueryResults.at(1);
                if (!serialized.empty()) {
                    TStringInput in{serialized};
                    ::LoadMany(&in, SnapshotTxn_, LastDeletedWalTxn_, LastSnapshottedWalTxn_);
                    LastWrittenWalTxn_ = LastSnapshottedWalTxn_;
                    const auto now = TInstant::Now();
                    for (const auto& [numId, _]: LastSnapshottedWalTxn_) {
                        LastWrittenWalTime_[numId] = now;
                    }
                } else {
                    ResetLastDeletedWalTxnMetric1_->Inc();
                }
            }

            if (LastDeletedWalTxn_ > WalTxn_) {
                MON_WARN(
                    Wal,
                    LOG_P << "wal status is inconsistent, wal will not be loaded: "
                          << "txn=" << WalTxn_ << ", "
                          << "last_deleted_txn=" << LastDeletedWalTxn_);
                LastDeletedWalTxn_ = WalTxn_;
                LastSnapshottedWalTxn_ = {};
                LastWrittenWalTime_ = {};
                LastWrittenWalTxn_ = {};
                ResetLastDeletedWalTxnMetric2_->Inc();
                Init3Done();
                return;
            }

            if (WalTxn_ - LastDeletedWalTxn_ > 150000) {
                // no snapshots were created, and current txn is way too big. This means that memstore
                // was working for some time with snapshotting disabled. We don't want to load snapshots
                // because it may take significant amount of time.
                // TODO: be smart about this, we need to list all wal files and only load recent ones.
                MON_WARN(
                    Wal,
                    LOG_P << "wal is too large and will not be loaded: "
                          << "txn=" << WalTxn_ << ", "
                          << "last_deleted_txn=" << LastDeletedWalTxn_);
                LastDeletedWalTxn_ = WalTxn_;
                LastSnapshottedWalTxn_ = {};
                LastWrittenWalTime_ = {};
                LastWrittenWalTxn_ = {};
                ResetLastDeletedWalTxnMetric3_->Inc();
                Init3Done();
                return;
            }

            // Start loading wal records
            {
                InitCurrentWalFile_ = LastDeletedWalTxn_ + 1;
                MON_DEBUG(Wal, LOG_P << "will read wal between txn " << InitCurrentWalFile_ << " and " << WalTxn_);
                Init2ReadWalFile();
            }
        }
    }

    void Init2ReadWalFile() {
        if (WalState_ != StateNotInited || InitIsReadWal_) {
            return;
        }

        if (InitCurrentWalFile_ > WalTxn_) {
            Init3Done();
            return;
        }

        if (IndexLimiter_.Locked() || !IndexGlobalLimiter_->CanAddIndexMessage()) {
            Schedule(INIT_READ_WAL_FILE_PERIOD, new TEvRetryWalRead{});
            return;
        }

        MON_DEBUG(Wal, LOG_P << "reading wal at txn " << InitCurrentWalFile_ << "." << InitCurrentWalFileIndex_);
        InitIsReadWal_ = true;
        TKikimrKvBatchRequest batch{TabletId_};
        batch.ReadFile(WalFilename(NodeId_, InitCurrentWalFile_, InitCurrentWalFileIndex_, false, false));
        batch.ReadFile(WalFilename(NodeId_, InitCurrentWalFile_, InitCurrentWalFileIndex_, true, false));
        Send(Client_, NKv::TEvents::Batch(std::move(batch)).Release(), 0, /* cookie = */ KvCookieWalRead);
    }

    void Init2ReadWalFileDone(IEventHandle& handle) {
        Y_VERIFY(InitIsReadWal_);
        auto ev = handle.Get<NKv::TEvents::TBatchResponse>();
        InitIsReadWal_ = false;

        if (ev->Fail()) {
            MON_ERROR(Wal, LOG_P << "reading wal error: " << ev->Error().Message());
            Schedule(Config_.FetchTxnBackoff, new TEvRetryWalRead{});
            return;
        } else {
            if (!ev->Value().ReadFileQueryResults.at(0).empty()) {
                const auto& chunk = ev->Value().ReadFileQueryResults.at(0);
                InitCurrentWalFileContents_ += chunk;
                WalMemorySizeBytesMetric_->Add(chunk.size());
                InitCurrentWalFileIndex_ += 1;
                Init2ReadWalFile();
            } else {
                const auto& chunk = ev->Value().ReadFileQueryResults.at(1);
                InitCurrentWalFileContents_ += chunk;
                WalMemorySizeBytesMetric_->Add(chunk.size());

                // parse file
                if (!InitCurrentWalFileContents_.empty()) {
                    auto indexEv = ReadCurWalFileContents();
                    if (indexEv) {
                        IndexGlobalLimiter_->AddIndexMessageCount(indexEv->Requests.size());
                        Send(Config_.Index, indexEv.Release());
                    }
                } else {
                    MON_WARN(Wal, LOG_P << "wal at txn " << InitCurrentWalFile_ << " is empty");
                }

                InitCurrentWalFile_ += 1;
                InitCurrentWalFileIndex_ = 0;
                WalMemorySizeBytesMetric_->Add(-static_cast<i64>(InitCurrentWalFileContents_.size()));
                InitCurrentWalFileContents_.clear();

                Init2ReadWalFile();
            }
        }
    }

    THolder<NIndex::TShardManagerEvents::TIndexBatch> ReadCurWalFileContents() {
        auto indexEv = MakeHolder<NIndex::TShardManagerEvents::TIndexBatch>(TLogId{TabletId_, InitCurrentWalFile_});
        size_t loaded = 0;

        TStringInput in{InitCurrentWalFileContents_};
        auto decodeIndex = [&in, this]() -> std::optional<NSlog::TSlogIndex> {
            std::optional<NSlog::TSlogIndex> ret{};
            try {
                ret = std::move(NSlog::DecodeIndex(&in));
            } catch (NSlog::TDecodeError& err) {
                MON_ERROR(Wal, LOG_P << "decoding wal index error: " << err.what());
            }
            return ret;
        };
        auto index = decodeIndex();

        if (index) {
            for (size_t i = 0; i < index.value().Size(); ++i) {
                TNumId numId = index.value().GetNumId(i);
                auto metaSize = index.value().GetMetaSizeBytes(i);
                auto dataSize = index.value().GetDataSizeBytes(i);

                if (auto it = LastSnapshottedWalTxn_.find(numId);
                        it != LastSnapshottedWalTxn_.end() && InitCurrentWalFile_ <= it->second) {
                    // this data is already snapshotted
                    MON_DEBUG(Wal,
                            LOG_P << "this data is already snapshotted: " << numId << " in txn "
                                    << InitCurrentWalFile_);
                    auto toSkip = metaSize + dataSize;
                    Y_VERIFY(in.Skip(toSkip) == toSkip, "malformed wal record txn=%lu", InitCurrentWalFile_);
                    continue;
                }

                TString meta(metaSize, '\0');
                Y_VERIFY(in.Load(meta.begin(), metaSize) == metaSize,
                        "malformed wal record txn=%lu",
                        InitCurrentWalFile_);

                TString data(dataSize, '\0');
                Y_VERIFY(in.Load(data.begin(), dataSize) == dataSize,
                        "malformed wal record txn=%lu",
                        InitCurrentWalFile_);

                IndexLimiter_.Inc(data.size() + meta.size());

                NIndex::TShardAddPointRequests& requests = indexEv->Requests[numId];
                if (requests.NumId == 0) {
                    requests.NumId = numId;
                    requests.ShardKey = {}; // no way to restore shard key on this level
                }
                requests.Requests.push_back({std::move(meta), std::move(data), {}, 0});

                if (!LastSnapshottedWalTxn_.contains(numId)) {
                    Y_VERIFY(InitCurrentWalFile_ > 0);
                    LastSnapshottedWalTxn_[numId] = InitCurrentWalFile_ - 1;
                }
                LastWrittenWalTxn_[numId] = InitCurrentWalFile_;
                LastWrittenWalTime_[numId] = TInstant::Now();

                loaded++;
            }
        }
        MON_DEBUG(Wal, LOG_P << "loaded " << loaded << " write request(s)");

        if (loaded == 0) {
            WalTxnToDelete_.push_back(InitCurrentWalFile_);
            return nullptr;
        }
        return indexEv;
    }

    void Init3Done() {
        WalMemorySizeBytesMetric_->Add(-static_cast<i64>(InitCurrentWalFileContents_.size()));
        InitCurrentWalFileContents_ = TString();
        WalState_ = StateIdle;
        SnapshotState_ = StateIdle;

        UpdateLastDeletedWalTxn();
        if (LastDeletedWalTxn_ + 10000 < WalTxn_) {
            MON_WARN(
                    Wal,
                    LOG_P << "suspicious wal state: "
                    << "txn=" << WalTxn_ << ", "
                    << "last_deleted_txn=" << LastDeletedWalTxn_);
        }

        Schedule(HalfJitter_(Config_.ClearWalFilesPeriod), new TEvClearWalFiles{});
        if (WalManager_) {
            Send(WalManager_, new TWalEvents::TInitDone(TabletId_));
        }
        WalProgress();
        SnapshotProgress();
    }

private:
    // Wal writer

    struct TWalWriteRequest {
        TNumId NumId;
        TShardKey ShardKey;
        TString Meta;
        TString Data;
        TActorId ReplyTo;
        ui64 Cookie;
        TInstant ProcessingStartTime;

        ui64 SizeBytes() const noexcept {
            return Meta.size() + Data.size();
        }
    };

    class TIndexLimiter {
    public:
        explicit TIndexLimiter(NMonitoring::TMetricRegistry& registry) {
            IndexInflightBytes_ = registry.IntGauge({{"sensor", "wal.indexInflightBytes"}});
        }

        void Inc(ui64 delta) {
            InflightBytes_ += delta;
            IndexInflightBytes_->Add(delta);
            if (InflightBytes_ >= MAX_INDEX_INFLIGHT) {
                Locked_ = true;
            }
        }

        void Dec(ui64 delta) {
            InflightBytes_ -= delta;
            IndexInflightBytes_->Add(- (i64) delta);
            if (InflightBytes_ < MAX_INDEX_INFLIGHT / 2) {
                Locked_ = false;
            }
        }

        bool Locked() const {
            return Locked_;
        }

    private:
        ui64 InflightBytes_{0};
        bool Locked_{false};

        NMonitoring::IIntGauge* IndexInflightBytes_;
    };

    EState WalState_ = StateNotInited;
    TTxn WalTxn_ = 0;

    size_t WalQueueSize_ = 0;
    TDeque<TWalWriteRequest> WalQueue_;

    size_t WalStageSize_ = 0;
    TDeque<TRope> WalStage_;
    TVector<TWalWriteRequest> WalStageRequests_ = {};
    ui64 WalStageTxn_ = 0;

    TIndexLimiter IndexLimiter_;

    void OnIndexDone(const NIndex::TShardEvents::TIndexDone::TPtr& ev) {
        bool locked = IndexLimiter_.Locked();

        IndexLimiter_.Dec(ev->Get()->WrittenBytes);
        TActivationContext::Send(ev->Forward(Config_.Index));

        if (locked && !IndexLimiter_.Locked()) {
            if (WalState_ == EState::StateNotInited) {
                Init2ReadWalFile();
            } else {
                WalProgress();
            }
        }
    }

    void OnSetDispatcher(TWalEvents::TSetDispatcher::TPtr& handle) {
        WalManager_ = std::move(handle->Get()->Dispatcher);
        switch (WalState_) {
            case StateNotInited:
                break;
            case StateIdle:
            case StateWrite:
            case StateWriteInProgress:
            case StateWriteDone:
            case StateFinalize:
            case StateFinalizeInProgress:
            case StateFinalizeDone:
                Send(WalManager_, new TWalEvents::TInitDone(TabletId_));
                break;
        }
    }

    void OnAddLogRecord(TWalEvents::TAddLogRecord::TPtr& handle) {
        auto ev = handle->Get();
        ui64 cookie = handle->Cookie;

        if (Closed_) {
            MON_DEBUG(Wal, LOG_P << "rejecting request: write queue is closed");
            RejectedEvents_->Inc();
            Send(handle->Sender, new TWalEvents::TAddLogRecordResult{
                TApiError{grpc::UNAVAILABLE, "write queue is closed"}}, 0, cookie);
            return;
        }

        if (!IndexGlobalLimiter_->CanAddIndexMessage()) {
            MON_DEBUG(Wal, LOG_P << "rejecting request: global write inflight data limit exceeded");
            RejectedEvents_->Inc();
            Send(handle->Sender, new TWalEvents::TAddLogRecordResult{
                    TApiError{grpc::RESOURCE_EXHAUSTED, "global write inflight data limit exceeded"}}, 0, cookie);
            return;
        }

        auto chunkSize = ev->Data.size() + ev->MetaData.size();

        if (WalQueueSize_ + chunkSize > Config_.MaxQueueSize) {
            MON_DEBUG(Wal, LOG_P << "rejecting request: write queue is full");
            RejectedEvents_->Inc();
            Send(handle->Sender, new TWalEvents::TAddLogRecordResult{
                TApiError{grpc::RESOURCE_EXHAUSTED, "write queue is full"}}, 0, cookie);
            return;
        }

        if (Config_.Limiter && !Config_.Limiter->TryAdd(chunkSize)) {
            MON_DEBUG(Wal, LOG_P << "rejecting request: global queue limit exceeded");
            RejectedEvents_->Inc();
            Send(handle->Sender, new TWalEvents::TAddLogRecordResult{
                TApiError{grpc::RESOURCE_EXHAUSTED, "global queue limit exceeded"}}, 0, cookie);
            return;
        }

        GlobalQueueSizeMetric_->Add(chunkSize);
        QueueSizeMetric_->Add(chunkSize);

        WalQueueSize_ += chunkSize;
        WalQueue_.push_back(TWalWriteRequest{
            ev->NumId,
            std::move(ev->ShardKey),
            std::move(ev->MetaData),
            std::move(ev->Data),
            handle->Sender,
            cookie,
            TInstant::Now()});

        WalProgress();
        MaybePassAway();
    }

    void WalWriteDone(IEventHandle& handle) {
        Y_VERIFY(WalState_ == StateWriteInProgress);

        auto ev = handle.Get<NKv::TEvents::TWriteFileResponse>();

        if (ev->Success()) {
            MON_DEBUG(Wal, LOG_P << "write wal success");

            WriteRpsMetric_->Inc();
            WriteSizeRpsMetric_->Add(WalStage_.front().GetSize());

            WalState_ = StateWriteDone;
            WalProgress();
        } else {
            MON_ERROR(Wal, LOG_P << "write wal error: " << ev->Error().Message());
            WriteErrRpsMetric_->Inc();
            Schedule(Config_.WriteBackoff, new TEvRetryWalWrite{});
        }
    }

    void WalWriteRetry() {
        Y_VERIFY(WalState_ == StateWriteInProgress);

        WalState_ = StateWrite;
        WalProgress();
    }

    void WalFinalizeDone(IEventHandle& handle) {
        Y_VERIFY(WalState_ == StateFinalizeInProgress);

        auto ev = handle.Get<NKv::TEvents::TBatchResponse>();

        if (ev->Success()) {
            MON_DEBUG(Wal, LOG_P << "wal finalization success");

            WriteRpsMetric_->Inc();
            WriteSizeRpsMetric_->Add(sizeof(WalTxn_));

            WalState_ = StateFinalizeDone;
            WalProgress();
        } else {
            MON_ERROR(Wal, LOG_P << "wal finalization error: " << ev->Error().Message());
            WriteErrRpsMetric_->Inc();
            Schedule(Config_.WriteBackoff, new TEvRetryWalFinalize{});
        }
    }

    void WalFinalizeRetry() {
        Y_VERIFY(WalState_ == StateFinalizeInProgress);

        WalState_ = StateFinalize;
        WalProgress();
    }

    void WalProgress() {
        switch (WalState_) {
            case StateNotInited:
                break;
            case StateIdle:
                WalOnIdle();
                break;
            case StateWrite:
                WalOnWrite();
                break;
            case StateWriteInProgress:
                break;
            case StateWriteDone:
                WalOnWriteDone();
                break;
            case StateFinalize:
                WalOnFinalize();
                break;
            case StateFinalizeInProgress:
                break;
            case StateFinalizeDone:
                WalOnFinalizeDone();
                break;
        }
    }

    void WalOnIdle() {
        WalStage_.clear();
        WalStageRequests_.clear();
        WalStageSize_ = 0;
        WalStageTxn_ = 0;

        if (WalQueue_.empty()) {
            return;
        }

        // Refill stage
        {
            WalTxn_ += 1;
            WalTxnMetric_->Set(WalTxn_);
            ReportUnsnapshotCount();

            TRope contents;
            NSlog::TSlogIndex index{WalQueue_.size()};

            while (!WalQueue_.empty()) {
                auto req = std::move(WalQueue_.front());
                WalQueue_.pop_front();
                WalStageSize_ += req.SizeBytes();
                index.Add(req.NumId, req.Meta.size(), req.Data.size());
                contents.Insert(contents.End(), TRope{req.Meta});
                contents.Insert(contents.End(), TRope{req.Data});
                WalStageRequests_.push_back(std::move(req));
            }

            TStringStream serializedIndex;
            NSlog::EncodeIndex(&serializedIndex, index);
            contents.Insert(contents.Begin(), TRope(serializedIndex.Str()));

            while (!contents.IsEmpty()) {
                WalStage_.push_back(contents.Extract(
                    contents.Begin(),
                    contents.GetSize() <= Config_.BatchSize ? contents.End() : contents.Position(Config_.BatchSize)));
            }

            WalStage_.shrink_to_fit();
            WalQueue_.shrink_to_fit();
        }

        // Start writing stage
        {
            WalState_ = StateWrite;
            WalProgress();
        }
    }

    void WalOnWrite() {
        Y_VERIFY(!WalStage_.empty());
        auto name = WalFilename(
            NodeId_, WalTxn_, WalStageTxn_, WalStage_.size() == 1, true);
        MON_DEBUG(Wal, LOG_P << "scheduling write for file " << name);
        Send(
            Client_,
            NKv::TEvents::WriteFile(
                TabletId_,
                name,
                WalStage_.front().ConvertToString(),
                Config_.WriteTimeout.ToDeadLine(),
                true).Release(),
            /* flags = */ 0,
            /* cookie = */ KvCookieWalWrite);

        WalState_ = StateWriteInProgress;
    }

    void WalOnWriteDone() {
        Y_VERIFY(!WalStage_.empty());
        WalStage_.pop_front();

        if (!WalStage_.empty()) {
            WalStageTxn_ += 1;
            WalState_ = StateWrite;
            WalProgress();
        } else {
            WalState_ = StateFinalize;
            WalProgress();
        }
    }

    void WalOnFinalize() {
        TString serialized;
        TStringOutput out{serialized};
        ::Save(&out, WalTxn_);

        auto deadline = TDuration::Seconds(60).ToDeadLine();
        auto batch = TKikimrKvBatchRequest(TabletId_, deadline);
        auto tmpPrefix = WalFilenamePrefix(NodeId_, WalTxn_, true);
        auto logPrefix = WalFilenamePrefix(NodeId_, WalTxn_, false);
        batch.CopyFiles({tmpPrefix, tmpPrefix + "~"}, tmpPrefix, logPrefix);
        batch.RemoveFiles({tmpPrefix, tmpPrefix + "~"});
        batch.WriteFile(CurrentWalTxnFile_, serialized);
        MON_DEBUG(Wal, LOG_P << "scheduling finalization for file " << tmpPrefix << "*");
        Send(Client_, NKv::TEvents::Batch(std::move(batch), true).Release(), 0, KvCookieWalFinalize);

        WalState_ = StateFinalizeInProgress;
    }

    void WalOnFinalizeDone() {
        WalQueueSize_ -= WalStageSize_;
        if (Config_.Limiter) {
            Config_.Limiter->Sub(WalStageSize_);
        }
        GlobalQueueSizeMetric_->Add(-WalStageSize_);
        QueueSizeMetric_->Add(-WalStageSize_);

        auto now = TInstant::Now();
        auto indexEv = MakeHolder<NIndex::TShardManagerEvents::TIndexBatch>(TLogId{TabletId_, WalTxn_});
        indexEv->Requests.reserve(WalStageRequests_.size());
        for (auto& req: WalStageRequests_) {
            EventProcessingTime_->Record((now - req.ProcessingStartTime).MilliSeconds());
            Send(req.ReplyTo, new TWalEvents::TAddLogRecordResult{TLogId{TabletId_, WalTxn_}}, 0, req.Cookie);

            NIndex::TShardAddPointRequests& requests = indexEv->Requests[req.NumId];
            if (requests.NumId == 0) {
                requests.NumId = req.NumId;
                requests.ShardKey = std::move(req.ShardKey);
            }
            IndexLimiter_.Inc(req.SizeBytes());
            requests.Requests.emplace_back(NIndex::TAddPointRequest{
                std::move(req.Meta),
                std::move(req.Data),
                req.ReplyTo,
                req.Cookie,
            });

            if (!LastSnapshottedWalTxn_.contains(req.NumId)) {
                Y_VERIFY(WalTxn_ > 0);
                LastSnapshottedWalTxn_[req.NumId] = WalTxn_ - 1;
            }
            LastWrittenWalTxn_[req.NumId] = WalTxn_;
            LastWrittenWalTime_[req.NumId] = TInstant::Now();
        }
        IndexGlobalLimiter_->AddIndexMessageCount(indexEv->Requests.size());
        Send(Config_.Index, indexEv.Release());

        WalState_ = StateIdle;
        WalProgress();
        MaybePassAway();
    }

private:
    struct TSnapshotWriteRequest {
        TNumId NumId;
        TString Meta;
        TString Data;
        TActorId ReplyTo;
        TTxn LatestWalTxn;
    };

    // Snapshot writer
    EState SnapshotState_ = StateNotInited;
    TTxn SnapshotTxn_ = 0;
    TTxn LastDeletedWalTxn_ = 0;
    // For each shard, we keep track of which wal record was last snapshotted.
    // This way when loading wal files, we know which records should be restored, and which should not.
    THashMap<TNumId, TTxn> LastSnapshottedWalTxn_;
    // For each shard, we also keep track of which wal record was last written.
    // This is necessary to clean `LastSnapshottedWalTxn_`. I.e. if we have just deleted wal record `5`, and we know
    // that shard `1` did not appear in any wal record after `5`, then we can remove shard `1` from both
    // `LastSnapshottedWalTxn_` and `LastWrittenWalTxn_`.
    // We do not persist this map because its state can be restored from wal records.
    THashMap<TNumId, TTxn> LastWrittenWalTxn_;
    THashMap<TNumId, TInstant> LastWrittenWalTime_;

    TDeque<TSnapshotWriteRequest> SnapshotQueue_;

    size_t SnapshotStageSize_ = 0;
    TDeque<TRope> SnapshotStage_;
    TVector<TSnapshotWriteRequest> SnapshotStageRequests_;
    ui64 SnapshotStageTxn_ = 0;
    TVector<TTxn> WalTxnToDelete_;
    size_t WalTxnDelCount_{0};

    void OnSnapshot(TWalEvents::TSnapshot::TPtr& handle) {
        auto ev = handle->Get();

        auto chunkSize = ev->Meta.size() + ev->Data.size();

        GlobalSnapshotQueueSizeMetric_->Add(chunkSize);
        SnapshotQueueSizeMetric_->Add(chunkSize);
        ShapshotStartRpsMetric_->Inc();

        SnapshotQueue_.push_back(TSnapshotWriteRequest{
            ev->NumId,
            std::move(ev->Meta),
            std::move(ev->Data),
            ev->Subscriber,
            ev->LogId.Txn});

        SnapshotProgress();
        MaybePassAway();
    }

    void SnapshotWriteDone(IEventHandle& handle) {
        SnapshotWriteDoneRpsMetric_->Inc();

        Y_VERIFY(SnapshotState_ == StateWriteInProgress);

        auto ev = handle.Get<NKv::TEvents::TWriteFileResponse>();

        if (ev->Success()) {
            MON_DEBUG(Wal, LOG_P << "snapshot write success");

            WriteRpsMetric_->Inc();
            WriteSizeRpsMetric_->Add(SnapshotStage_.front().GetSize());

            SnapshotState_ = StateWriteDone;
            SnapshotProgress();
        } else {
            MON_ERROR(Wal, LOG_P << "snapshot write error: " << ev->Error().Message());
            WriteErrRpsMetric_->Inc();
            Schedule(Config_.WriteBackoff, new TEvRetrySnapshotWrite{});
        }
    }

    void SnapshotWriteRetry() {
        Y_VERIFY(SnapshotState_ == StateWriteInProgress);

        SnapshotState_ = StateWrite;
        SnapshotProgress();
    }

    void SnapshotFinalizeDone(IEventHandle& handle) {
        Y_VERIFY(SnapshotState_ == StateFinalizeInProgress);

        auto ev = handle.Get<NKv::TEvents::TBatchResponse>();

        if (ev->Success()) {
            MON_DEBUG(Wal, LOG_P << "snapshot finalization success");

            WriteRpsMetric_->Inc();
            WriteSizeRpsMetric_->Add(sizeof(SnapshotTxn_));
            ShapshotFinalizeDoneRpsMetric_->Inc();

            SnapshotState_ = StateFinalizeDone;
            SnapshotProgress();
        } else {
            MON_ERROR(Wal, LOG_P << "snapshot finalization error: " << ev->Error().Message());
            WriteErrRpsMetric_->Inc();
            Schedule(Config_.WriteBackoff, new TEvRetrySnapshotFinalize{});
        }
    }

    void SnapshotFinalizeRetry() {
        Y_VERIFY(SnapshotState_ == StateFinalizeInProgress);

        SnapshotState_ = StateFinalize;
        SnapshotProgress();
    }

    void SnapshotProgress() {
        switch (SnapshotState_) {
            case StateNotInited:
                break;
            case StateIdle:
                SnapshotOnIdle();
                break;
            case StateWrite:
                SnapshotOnWrite();
                break;
            case StateWriteInProgress:
                break;
            case StateWriteDone:
                SnapshotOnWriteDone();
                break;
            case StateFinalize:
                SnapshotOnFinalize();
                break;
            case StateFinalizeInProgress:
                break;
            case StateFinalizeDone:
                SnapshotOnFinalizeDone();
                break;
        }
    }

    void SnapshotOnIdle() {
        SnapshotOnIdleRpsMetric_->Inc();

        SnapshotStage_.clear();
        SnapshotStageRequests_.clear();
        SnapshotStageSize_ = 0;
        SnapshotStageTxn_ = 0;

        if (SnapshotQueue_.empty()) {
            return;
        }

        // Refill stage
        {
            SnapshotTxn_ += 1;

            TRope contents;
            NSlog::TSlogIndex index{SnapshotQueue_.size()};

            while (!SnapshotQueue_.empty()) {
                auto req = std::move(SnapshotQueue_.front());
                SnapshotQueue_.pop_front();
                SnapshotStageSize_ += req.Meta.size() + req.Data.size();
                index.Add(req.NumId, req.Meta.size(), req.Data.size());
                contents.Insert(contents.End(), TRope{req.Meta});
                contents.Insert(contents.End(), TRope{req.Data});
                if (auto it = LastSnapshottedWalTxn_.find(req.NumId); it != LastSnapshottedWalTxn_.end()) {
                    it->second = req.LatestWalTxn;
                } else {
                    MON_WARN(Wal, LOG_P << "got snapshot for an unknown shard num_id=" << req.NumId);
                }
                SnapshotStageRequests_.push_back(std::move(req));
            }

            TStringStream serializedIndex;
            NSlog::EncodeIndex(&serializedIndex, index);
            contents.Insert(contents.Begin(), TRope(serializedIndex.Str()));

            while (!contents.IsEmpty()) {
                SnapshotStage_.push_back(contents.Extract(
                    contents.Begin(),
                    contents.GetSize() <= Config_.BatchSize ? contents.End() : contents.Position(Config_.BatchSize)));
            }

            SnapshotStage_.shrink_to_fit();
            SnapshotQueue_.shrink_to_fit();
        }

        // Start writing stage
        {
            SnapshotState_ = StateWrite;
            SnapshotProgress();
        }
    }

    void SnapshotOnWrite() {
        SnapshotOnWriteRpsMetric_->Inc();

        Y_VERIFY(!SnapshotStage_.empty());
        auto name = SnapshotFilename(
            NodeId_, SnapshotTxn_, SnapshotStageTxn_, SnapshotStage_.size() == 1, true);
        MON_DEBUG(Wal, LOG_P << "scheduling write for file " << name);
        Send(
            Client_,
            NKv::TEvents::WriteFile(
                TabletId_,
                name,
                SnapshotStage_.front().ConvertToString(),
                Config_.WriteTimeout.ToDeadLine(),
                true).Release(),
            /* flags = */ 0,
            /* cookie = */ KvCookieSnapshotWrite);

        SnapshotState_ = StateWriteInProgress;
    }

    void SnapshotOnWriteDone() {
        SnapshotOnWriteDoneRpsMetric_->Inc();

        Y_VERIFY(!SnapshotStage_.empty());
        SnapshotStage_.pop_front();

        if (!SnapshotStage_.empty()) {
            SnapshotStageTxn_ += 1;
            SnapshotState_ = StateWrite;
        } else {
            UpdateLastDeletedWalTxn();
            SnapshotState_ = StateFinalize;
        }
        SnapshotProgress();
    }

    // Find the latest wal txn that should be deleted
    TTxn CalcLastDeletedWalTxn() const {
        TTxn lastDeletedWalTxn{LastDeletedWalTxn_};
        if (!LastSnapshottedWalTxn_.empty()) {
            TTxn txnToDelete;
            bool foundTxnToDelete = false;
            for (auto [numId, txn]: LastSnapshottedWalTxn_) {
                Y_VERIFY(LastWrittenWalTxn_.contains(numId));
                if (LastWrittenWalTxn_.at(numId) > txn) {
                    if (foundTxnToDelete) {
                        txnToDelete = Min(txnToDelete, txn);
                    } else {
                        txnToDelete = txn;
                    }
                    foundTxnToDelete = true;
                }
            }
            if (foundTxnToDelete) {
                lastDeletedWalTxn = txnToDelete;
            } else {
                // We've successfully snapshotted everything there is to snapshot, we can drop the whole wal now
                for (auto [numId, txn]: LastSnapshottedWalTxn_) {
                    lastDeletedWalTxn = Max(lastDeletedWalTxn, txn);
                }
            }
        }
        return lastDeletedWalTxn;
    }

    void UpdateLastDeletedWalTxn() {
        const auto lastDeletedWalTxn = CalcLastDeletedWalTxn();
        if (lastDeletedWalTxn != LastDeletedWalTxn_) {
            LastDeletedWalTxn_ = lastDeletedWalTxn;

            // Clean up `LastSnapshottedWalTxn_` and `LastWrittenWalTxn_`
            for (auto it = LastSnapshottedWalTxn_.begin(); it != LastSnapshottedWalTxn_.end(); ) {
                auto [numId, txn] = *it;
                if (txn <= LastDeletedWalTxn_ && LastWrittenWalTxn_[numId] == txn) {
                    LastSnapshottedWalTxn_.erase(it++);
                    LastWrittenWalTime_.erase(numId);
                    LastWrittenWalTxn_.erase(numId);
                } else {
                    it++;
                }
            }
        }
        LastDeletedWalTxnMetric_->Set(LastDeletedWalTxn_);
        ReportUnsnapshotCount();
    }

    void ReportUnsnapshotCount(bool isPoisoned = false) {
        const i64 newValue = isPoisoned ? 0 : static_cast<i64>(WalTxn_) - static_cast<i64>(LastDeletedWalTxn_);
        UnsnapshotTxnCountMetric_->Add(newValue - LastUnsnapshotCount_);
        LastUnsnapshotCount_ = newValue;
        UnsnapshotTxnTabletMetric_->Set(newValue);
    }

    void SnapshotOnFinalize() {
        SnapshotOnFinalizeRpsMetric_->Inc();

        TString serialized;
        TStringOutput out{serialized};
        ::SaveMany(&out, SnapshotTxn_, LastDeletedWalTxn_, LastSnapshottedWalTxn_);

        auto deadline = TDuration::Seconds(60).ToDeadLine();
        auto batch = TKikimrKvBatchRequest(TabletId_, deadline);
        auto tmpPrefix = SnapshotFilenamePrefix(NodeId_, SnapshotTxn_, true);
        auto logPrefix = SnapshotFilenamePrefix(NodeId_, SnapshotTxn_, false);
        batch.CopyFiles({tmpPrefix, tmpPrefix + "~"}, tmpPrefix, logPrefix);
        batch.RemoveFiles({tmpPrefix, tmpPrefix + "~"});
        batch.WriteFile(CurrentSnapshotTxnFile_, serialized);
        auto walPrefixBegin = WalFilenamePrefix(NodeId_, 0, false);
        auto walPrefixEnd = WalFilenamePrefix(NodeId_, LastDeletedWalTxn_, false);
        batch.RemoveFiles({walPrefixBegin, walPrefixEnd + "~"});
        MON_DEBUG(Wal, LOG_P << "scheduling finalization for file " << tmpPrefix << "*");
        Send(Client_, NKv::TEvents::Batch(std::move(batch), true).Release(), 0, KvCookieSnapshotFinalize);

        SnapshotState_ = StateFinalizeInProgress;
    }

    void ClearWalFiles() {
        const auto now = TInstant::Now();
        for (auto& [numId, txn]: LastSnapshottedWalTxn_) {
            Y_VERIFY(LastWrittenWalTxn_.contains(numId));
            const auto it = LastWrittenWalTime_.find(numId);
            Y_VERIFY(it != LastWrittenWalTime_.end());
            const auto duration = now - it->second;
            if (duration > TDuration::Minutes(60)) {
                txn = LastWrittenWalTxn_[numId];
            }
        }
        UpdateLastDeletedWalTxn();

        auto deadline = MAX_CLEAR_REQUEST_DURATION.ToDeadLine();
        auto batch = TKikimrKvBatchRequest(TabletId_, deadline);
        auto walPrefixBegin = WalFilenamePrefix(NodeId_, 0, false);
        auto walPrefixEnd = WalFilenamePrefix(NodeId_, LastDeletedWalTxn_, false);
        batch.RemoveFiles({walPrefixBegin, walPrefixEnd + "~"});

        if (!WalTxnToDelete_.empty() && WalTxnDelCount_ == 0) {
            constexpr size_t MaxDeleteCount = 10'000;
            size_t delCount = 0;
            for (auto txn: WalTxnToDelete_) {
                batch.RemovePrefix(WalFilenamePrefix(NodeId_, txn, /* tmp: */ false));
                if (++delCount >= MaxDeleteCount) {
                    break;
                }
            }
            WalTxnDelCount_ = delCount;
        }

        Send(Client_, NKv::TEvents::Batch(std::move(batch), true).Release(), 0, KvCookieClearWalFiles);
        Schedule(HalfJitter_(Config_.ClearWalFilesPeriod), new TEvClearWalFiles{});
    }

    void ClearWalFilesDone(IEventHandle& handle) {
        auto ev = handle.Get<NKv::TEvents::TBatchResponse>();
        if (ev->Fail()) {
            MON_ERROR(Wal, LOG_P << "snapshot logs clear error: " << ev->Error().Message());
            ClearErrRpsMetric_->Inc();
        } else {
            if (WalTxnDelCount_ != 0) {
                WalTxnToDelete_.erase(WalTxnToDelete_.begin(), WalTxnToDelete_.begin() + WalTxnDelCount_);
                ClearWalCountMetric_->Add(WalTxnDelCount_);
                WalTxnDelCount_ = 0;
            }
        }
    }

    void SnapshotOnFinalizeDone() {
        GlobalSnapshotQueueSizeMetric_->Add(-SnapshotStageSize_);
        SnapshotQueueSizeMetric_->Add(-SnapshotStageSize_);
        ShapshotOnFinalizeDoneRpsMetric_->Inc();

        for (auto& req: SnapshotStageRequests_) {
            Send(req.ReplyTo, MakeHolder<TWalEvents::TSnapshotProcessed>());
        }

        SnapshotState_ = StateIdle;
        SnapshotProgress();
        MaybePassAway();
    }

private:
    NMonitoring::IIntGauge* GlobalQueueSizeMetric_;
    NMonitoring::IIntGauge* QueueSizeMetric_;
    NMonitoring::IIntGauge* GlobalSnapshotQueueSizeMetric_;
    NMonitoring::IIntGauge* SnapshotQueueSizeMetric_;
    NMonitoring::IRate* ShapshotStartRpsMetric_;
    NMonitoring::IRate* SnapshotOnIdleRpsMetric_;
    NMonitoring::IRate* SnapshotOnWriteRpsMetric_;
    NMonitoring::IRate* SnapshotWriteDoneRpsMetric_;
    NMonitoring::IRate* SnapshotOnWriteDoneRpsMetric_;
    NMonitoring::IRate* SnapshotOnFinalizeRpsMetric_;
    NMonitoring::IRate* ShapshotFinalizeDoneRpsMetric_;
    NMonitoring::IRate* ShapshotOnFinalizeDoneRpsMetric_;
    NMonitoring::IRate* WriteRpsMetric_;
    NMonitoring::IRate* WriteErrRpsMetric_;
    NMonitoring::IRate* WriteSizeRpsMetric_;
    NMonitoring::IRate* RejectedEvents_;
    NMonitoring::IHistogram* EventProcessingTime_;
    NMonitoring::IIntGauge* WalTxnMetric_;
    NMonitoring::IIntGauge* LastDeletedWalTxnMetric_;
    NMonitoring::IIntGauge* UnsnapshotTxnTabletMetric_;
    NMonitoring::IIntGauge* UnsnapshotTxnCountMetric_;
    i64 LastUnsnapshotCount_{0};
    NMonitoring::IIntGauge* ResetWalTxnMetric_;
    NMonitoring::IIntGauge* ResetLastDeletedWalTxnMetric1_;
    NMonitoring::IIntGauge* ResetLastDeletedWalTxnMetric2_;
    NMonitoring::IIntGauge* ResetLastDeletedWalTxnMetric3_;
    NMonitoring::IIntGauge* ClearWalCountMetric_;
    NMonitoring::IIntGauge* WalMemorySizeBytesMetric_;
    NMonitoring::IRate* ClearErrRpsMetric_;
    THalfJitter HalfJitter_;
    NActors::TActorId WalManager_;
    std::shared_ptr<IIndexWriteLimiter> IndexGlobalLimiter_;
};

} // namespace

std::unique_ptr<IActor> CreateWriteQueue(
        TActorId client,
        ui32 nodeId,
        TTabletId tabletId,
        TMaybe<TTxn> txn,
        TMaybe<TTxn> snapshotTxn,
        std::shared_ptr<IIndexWriteLimiter> indexWriteLimiter,
        NMonitoring::TMetricRegistry& registry,
        TWriteQueueConfig config)
{
    return std::make_unique<TWriteQueue>(
            client,
            nodeId,
            tabletId,
            txn,
            snapshotTxn,
            std::move(config),
            std::move(indexWriteLimiter),
            registry);
}

} // namespace NSolomon::NMemStore::NWal
