#pragma once

#include "wal_events.h"

#include <solomon/services/memstore/lib/types.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/limiter/limiter.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/size_literals.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NMemStore::NWal {

/**
 * Settings for write queue.
 */
struct TWriteQueueConfig {
    /**
     * Max queue size, in bytes.
     */
    size_t MaxQueueSize = 50_MB;

    /**
     * Maximal size of a single write request, in bytes. Files that are bigger than this size will be split in chunks.
     */
    size_t BatchSize = 20_MB;

    /**
     * Server timeout for write request.
     */
    TDuration WriteTimeout = TDuration::Seconds(3);

    /**
     * If write fails, schedule next retry no sooner than this time.
     *
     * Note that this setting does not affect retry parameters set in KV client.
     */
    TDuration WriteBackoff = TDuration::MilliSeconds(10);

    /**
     * If reading latest txn fails, schedule next retry no sooner than this time.
     *
     * Note that this setting does not affect retry parameters set in KV client.
     */
    TDuration FetchTxnBackoff = TDuration::MilliSeconds(1);

    /**
     * Global queue limiter. Can be shared between multiple queues to limit total stored size.
     */
    ILimiterPtr Limiter;

    /**
     * Write queue will relay data to index after adding it to the write ahead log.
     */
    NActors::TActorId Index;

    /**
     * Period between tries to delete unnecessary wal files on kv tablet.
     */
    TDuration ClearWalFilesPeriod = TDuration::Minutes(5);
};

/**
 * A queue that buffers writes to a KV tablet.
 *
 * @param client        id of the KV client used to communicate with tablet (see `NSolomon::NKv::CreateKvClientActor`).
 * @param nodeId        id of this MemStore instance.
 * @param tabletId      id of the KV tablet to write data into.
 * @param txn           initial transaction number. If not given, write queue will determine it automatically.
 *                      this parameter is used for unit testing only. When it is given, wal manager will not clean up
 *                      temporary wal files at startup.
 * @param snapshotTxn   initial snapshot transaction number. See `txn` above.
 * @param config        other settings for writer queue.
 */
std::unique_ptr<NActors::IActor> CreateWriteQueue(
        NActors::TActorId client,
        ui32 nodeId,
        TTabletId tabletId,
        TMaybe<TTxn> txn,
        TMaybe<TTxn> snapshotTxn,
        std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> indexWriteLimiter,
        NMonitoring::TMetricRegistry& registry,
        TWriteQueueConfig config = {});

} // namespace NSolomon::NMemStore::NWal
