#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/yasm/yasm.h>

#include <util/generic/string.h>
#include <util/string/builder.h>

using NSolomon::NMemStore::NYasm::TUgramAlignerBuilder;
using NSolomon::NMemStore::NYasm::TUgramBuilder;
using NSolomon::NMemStore::NYasm::TUgramAligner;
using NSolomon::NMemStore::NYasm::TBuckets;

class UgramAligner: public ::testing::Test {
};

TString BucketsToStr(const TBuckets& lhs) {
    TStringBuilder sb;
    sb << "{";

    for (size_t i = 0; i < lhs.size(); ++i) {
        sb << lhs[i].UpperBound << ": " << lhs[i].Value;
        if (i + 1 != lhs.size()) {
            sb << ", ";
        }
    }

    sb << "}";

    return sb;
}

void VerifyBucketsEq(const TBuckets& lhs, const TBuckets& rhs) {
    EXPECT_EQ(lhs.size(), rhs.size());

    for (size_t i = 0; i < lhs.size(); ++i) {
        EXPECT_EQ(lhs[i].UpperBound, rhs[i].UpperBound) << "i:" << i;
        EXPECT_EQ(lhs[i].Value, rhs[i].Value) << "i:" << i;
    }
}

double Next(double value) {
    return std::nextafter(value, std::numeric_limits<double>::max());
}

TEST_F(UgramAligner, BasicAlign) {
    {
        TUgramAlignerBuilder builder;

        TBuckets ex1{{0.0, 0u}, {0.5, 5u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex2{{0.0, 0u}, {0.5, 2u}, {1.0, 2u}, {2.0, 2u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex3{{0.0, 0u}, {2.0, 5u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex4{{0.0, 0u}, {0.5, 2u}, {1.0, 2u}, {2.0, 1u}, {4u, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex5{{0.0, 0u}, {0.5, 2u}, {2.0, 3u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        builder.AddBuckets(ex1);
        builder.AddBuckets(ex2);
        builder.AddBuckets(ex3);
        builder.AddBuckets(ex4);
        builder.AddBuckets(ex5);

        auto alignerOpt = builder.ReleaseAligner();
        ASSERT_TRUE(alignerOpt.has_value());
        auto aligner = std::move(alignerOpt.value());

        TBuckets r1{{0.0, 0u}, {0.5, 5u}, {1.0, 0u}, {2.0, 0u}, {4u, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r2{{0.0, 0u}, {0.5, 2u}, {1.0, 2u}, {2.0, 2u}, {4u, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r3{{0.0, 0u}, {0.5, 1u}, {1.0, 1u}, {2.0, 3u}, {4u, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r4{{0.0, 0u}, {0.5, 2u}, {1.0, 2u}, {2.0, 1u}, {4u, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r5{{0.0, 0u}, {0.5, 2u}, {1.0, 1u}, {2.0, 2u}, {4u, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        VerifyBucketsEq(aligner.Align(ex1), r1);
        VerifyBucketsEq(aligner.Align(ex2), r2);
        VerifyBucketsEq(aligner.Align(ex3), r3);
        VerifyBucketsEq(aligner.Align(ex4), r4);
        VerifyBucketsEq(aligner.Align(ex5), r5);
    }

    {
        TUgramAlignerBuilder builder;

        TBuckets ex1{{98.0, 0u}, {100, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex2{{96.0, 0u}, {98.0, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        builder.AddBuckets(ex1);
        builder.AddBuckets(ex2);

        auto alignerOpt = builder.ReleaseAligner();
        ASSERT_TRUE(alignerOpt.has_value());
        auto aligner = std::move(alignerOpt.value());

        TBuckets r1{{96.0, 0u}, {98.0, 0u}, {100.0, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r2{{96.0, 0u}, {98.0, 1u}, {100.0, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        VerifyBucketsEq(aligner.Align(ex1), r1);
        VerifyBucketsEq(aligner.Align(ex2), r2);
    }

    {
        TUgramAlignerBuilder builder;

        TBuckets ex2{{0.0, 0u}, {Next(0.0), 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets ex1{{0.0, 0u}, {Next(0.0), 1u}, {1.5, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        builder.AddBuckets(ex1);
        builder.AddBuckets(ex2);

        auto alignerOpt = builder.ReleaseAligner();
        ASSERT_TRUE(alignerOpt.has_value());
        auto aligner = std::move(alignerOpt.value());

        TBuckets r2{{0.0, 0u}, {Next(0.0), 1u}, {1.5, 0u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        TBuckets r1{{0.0, 0u}, {Next(0.0), 1u}, {1.5, 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};

        VerifyBucketsEq(aligner.Align(ex1), r1);
        VerifyBucketsEq(aligner.Align(ex2), r2);
    }

    {
        TUgramAlignerBuilder builder;

        TVector<TBuckets> exs;
        for (size_t i = 0; i < 10u; ++i) {
            auto ex = TBuckets{{(double)i * 10, 0u}, {(double)(i * 10 + 10), 1u}, {NMonitoring::HISTOGRAM_INF_BOUND}};
            builder.AddBuckets(ex);
            exs.push_back(std::move(ex));
        }

        auto alignerOpt = builder.ReleaseAligner();
        ASSERT_TRUE(alignerOpt.has_value());
        auto aligner = std::move(alignerOpt.value());

        TBuckets r;
        for (size_t i = 0; i < 11u; ++i) {
            r.push_back({(double)(i * 10), 0u});
        }
        r.push_back({NMonitoring::HISTOGRAM_INF_BOUND, 0u});

        for (size_t i = 0; i < 10; ++i) {
            r[i + 1].Value = 1;
            VerifyBucketsEq(aligner.Align(exs[i]), r);
            r[i + 1].Value = 0;
        }
    }

    {
        TUgramBuilder builder;
        builder.AddBucket(0.0, 0.0, 1u);

        auto act = builder.ReleaseBuckets();
        auto exp = TBuckets{{0.0, 0u}, {Next(0.0), 1u}, {NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
        VerifyBucketsEq(act, exp);
    }

}
