#include "yasm.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

#include <util/string/builder.h>

#include <limits>
#include <cmath>
#include <algorithm>
#include <type_traits>

using NMonitoring::TBucketValue;
using NMonitoring::TBucketBound;

namespace NSolomon::NMemStore::NYasm {

static constexpr NMonitoring::TBucketBound MAX_VALUE = std::numeric_limits<NMonitoring::TBucketBound>::max();

static TString BucketsToStr(const TBuckets& buckets) {
    TStringBuilder sb;
    sb << "{";
    for (size_t i = 0; i < buckets.size(); ++i) {
        sb << buckets[i].UpperBound << ": " << buckets[i].Value;
        if (i + 1 != buckets.size()) {
            sb << ", ";
        }
    }
    sb << "}";

    return sb;
}

static bool BucketsIsNormalized(const TBuckets& buckets) {
    return !buckets.empty() && buckets.back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND;
}
static bool BucketsIsDefined(const TBuckets& buckets) {
    if (buckets.empty()) {
        return true;
    }

    if (buckets.back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND) {
        return buckets.size() > 2u;
    } else {
        return buckets.size() > 1u;
    }
}
static bool BucketsIsEmpty(const TBuckets& buckets) {
    if (buckets.empty()) {
        return true;
    }

    if (buckets.back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND && buckets.size() == 1u) {
        return true;
    }

    return false;
}
static void VerifyUgramlike(const TBuckets& buckets) {
    if (buckets.empty()) {
        return;
    }

    if (buckets.back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND) {
        Y_ENSURE(buckets.size() > 2u, "invalid histogram: " << BucketsToStr(buckets));
    } else {
        Y_ENSURE(buckets.size() > 1u, "invalid histogram: " << BucketsToStr(buckets));
    }
}
static void VerifyHistogram(const TBuckets& buckets) {
    Y_ENSURE(!buckets.empty() || buckets.back().UpperBound != NMonitoring::HISTOGRAM_INF_BOUND,
            "invalid histogram: inf bucket not found: " << BucketsToStr(buckets));
    Y_ENSURE(buckets[0].Value == 0u, "invalid histogram: the inf bucket value must be 0: " << BucketsToStr(buckets));
    Y_ENSURE(buckets.back().Value == 0u, "invalid histogram: the first bucket's value must be 0: " << BucketsToStr(buckets));
    Y_ENSURE(buckets.size() < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, "invalid histogram: too many buckets " << buckets.size() << ": " << BucketsToStr(buckets));
    Y_ENSURE(buckets.size() != 2u, "invalid histogram: " << BucketsToStr(buckets));
    for (size_t i = 1u; i < buckets.size(); ++i) {
        if (buckets[i].UpperBound <= buckets[i - 1].UpperBound) {
            ythrow yexception() << "invalid histogram: unsorted buckets: " << BucketsToStr(buckets);
        }
    }
}

struct TUgramLikeBucket {
    TBucketBound LowerBound;
    TBucketBound UpperBound;
    TBucketValue Value;
};

class TUgramLikeInserter {
public:
    TUgramLikeInserter(TBuckets& buckets)
        : Buckets_(buckets)
    {
    }

    void AddBucket(TBucketBound lower, TBucketBound upper, TBucketValue value) {
        if (Buckets_.empty()) {
            AddInitialBucket(lower);
            AddBucketSimple(lower, upper, value);
            return;
        }

        auto lastUpper = LastUpper();
        auto lastLower = LastLower();

        Y_ENSURE(lastLower <= lower, "bucket breaks order: " << BucketsToStr(Buckets_) << " + {" << lower << ", " << upper << ", " << value << "}");

        if (lastLower == lower && lastUpper == upper) {
            LastValue() += value;
            return;
        }

        if (lastUpper <= lower) {
            if (lastUpper < lower) {
                AddBucketSimple(lower, 0u);
            }
            AddBucketSimple(lower, upper, value);
            return;
        }

        auto lastValue = LastValue();
        auto lastRange = lastUpper - lastLower;

        Buckets_.pop_back();

        if (lastLower < lower) {
            AddBucketSimple(lower, std::round(lastValue * (lower - lastLower) / lastRange));
        }

        if (upper <= lastUpper) {
            AddBucketSimple(lower, upper, std::round(value + lastValue * (upper - lower) / lastRange));
            if (upper < lastUpper) {
                AddBucketSimple(lastUpper, std::round(lastValue * (lastUpper - upper) / lastRange));
            }

            return;
        }

        double leftPart = lastValue * (upper - lastUpper) / lastRange;
        double rightPart = value * (upper - lastUpper) / (upper - lower);

        AddBucketSimple(lastUpper, std::round(leftPart + rightPart));
        AddBucketSimple(upper, std::round(value - rightPart));
    }

    // for std::back_inserter
    using value_type = TUgramLikeBucket;
    void push_back(TUgramLikeBucket bucket) {  // NOLINT
        AddBucket(bucket.LowerBound, bucket.UpperBound, bucket.Value);
    }

private:
    void AddInitialBucket(TBucketBound bound) {
        Y_VERIFY_DEBUG(Buckets_.empty());
        Buckets_.emplace_back(TBucket{bound, 0u});
    }

    void AddPointBucketSimple(TBucketBound bound, TBucketValue value) {
        Buckets_.emplace_back(TBucket{std::nextafter(bound, MAX_VALUE), value});
    }

    void AddBucketSimple(TBucketBound prev, TBucketBound upper, TBucketValue value) {
        if (upper == prev) {
            AddPointBucketSimple(prev, value);
        } else {
            Buckets_.emplace_back(TBucket{upper, value});
        }
    }

    void AddBucketSimple(TBucketBound bound, TBucketValue value) {
        Buckets_.emplace_back(TBucket{bound, value});
    }

    TBucketBound LastLower() const {
        if (Buckets_.size() == 1u) {
            return std::numeric_limits<TBucketBound>::lowest();
        }

        return LastBound(1u);
    }

    TBucketBound LastUpper() const {
        return LastBound(0u);
    }

    TBucketBound LastBound(size_t shift) const {
        Y_VERIFY_DEBUG(Buckets_.size() > shift);
        if (Buckets_.size() == 1u + shift) {
            return (Buckets_.rbegin() + shift)->UpperBound;
        }

        bool extend = (std::nextafter((Buckets_.rbegin() + shift + 1u)->UpperBound, MAX_VALUE) == (Buckets_.rbegin() + shift)->UpperBound);

        return (Buckets_.rbegin() + shift + extend)->UpperBound;
    }

    TBucketValue& LastValue(size_t shift = 0u) {
        Y_VERIFY_DEBUG(Buckets_.size() > shift);
        return (Buckets_.rbegin() + shift)->Value;
    }

private:
    TBuckets& Buckets_;
};

template <bool mut>
class TUgramLikeWrapperBase;

class TUgramLikeWrapper {
public:
    template <bool mut>
    using TBucketBoundRef = typename std::conditional<mut, TBucketBound&, const TBucketBound&>::type;

    template <bool mut>
    using TBucketValueRef = typename std::conditional<mut, TBucketValue&, const TBucketValue&>::type;

    template <bool mut>
    using TBucketRef = typename std::conditional<mut, TBuckets&, const TBuckets&>::type;

    template <bool mut>
    using TBucketPtr = typename std::conditional<mut, TBuckets*, const TBuckets*>::type;

    template <bool mut>
    class TIteratorBase;

    using TIterator = TIteratorBase<true>;
    using TConstIterator = TIteratorBase<false>;
};

template <bool mut>
class TUgramLikeWrapperBase: public TUgramLikeWrapper {
public:
    friend class TUgramLikeWrapperBase<!mut>;

    TUgramLikeWrapperBase(TBucketRef<mut> buckets)
        : Buckets_(&buckets)
    {
        VerifyUgramlike(buckets);
    }

    TUgramLikeWrapperBase(const TUgramLikeWrapperBase<true>& other)
        : Buckets_(other.Buckets_)
    {
    }

    TUgramLikeWrapperBase& operator=(const TUgramLikeWrapperBase<true>& other) {
        Buckets_ = other.Buckets_;
    }

    size_t Count() const {
        return Buckets_->empty() ? 0u: (Buckets_->size() - 1 - HasInfBucket());
    }

    TIteratorBase<mut> begin();

    TIteratorBase<mut> end();

    TConstIterator begin() const;

    TConstIterator end() const;

    TBucketBoundRef<mut> UpperBound(size_t i) {
        Y_VERIFY_DEBUG(0 <= i && i < Count());

        if (((*Buckets_))[i + 1].UpperBound == std::nextafter((*(Buckets_))[i].UpperBound, MAX_VALUE)) {
            return ((*Buckets_))[i].UpperBound;
        }

        return (*(Buckets_))[i + 1].UpperBound;
    }

    TBucketBoundRef<mut> LowerBound(size_t i) {
        Y_VERIFY_DEBUG(0 <= i && i < Count());

        if (i == 0) {
            return (*(Buckets_))[0].UpperBound;
        }

        return UpperBound(i - 1);
    }

    TBucketValueRef<mut> Value(size_t i) {
        Y_VERIFY_DEBUG(0 <= i && i < Count());
        return (*(Buckets_))[i + 1].Value;
    }

    TBucketBound UpperBound(size_t i) const {
        Y_VERIFY_DEBUG(0 <= i && i < Count());

        if (((*Buckets_))[i + 1].UpperBound == std::nextafter((*(Buckets_))[i].UpperBound, MAX_VALUE)) {
            return ((*Buckets_))[i].UpperBound;
        }

        return (*(Buckets_))[i + 1].UpperBound;
    }

    TBucketBound LowerBound(size_t i) const {
        Y_VERIFY_DEBUG(0 <= i && i < Count());

        if (i == 0) {
            return (*(Buckets_))[0].UpperBound;
        }

        return UpperBound(i - 1);
    }

    TBucketValue Value(size_t i) const {
        Y_VERIFY_DEBUG(0 <= i && i < Count());
        return (*(Buckets_))[i + 1].Value;
    }

private:
    bool HasInfBucket() const {
        return !Buckets_->empty() && Buckets_->back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND;
    }

private:
    TBucketPtr<mut> Buckets_;
};

template <bool mut>
class TUgramLikeWrapper::TIteratorBase {
public:
    friend class TUgramLikeWrapperBase<false>;
    friend class TUgramLikeWrapperBase<true>;
    friend class TIteratorBase<!mut>;

    TBucketBoundRef<mut> LowerBound() {
        return Ugram_.LowerBound(Index_);
    }

    TBucketBoundRef<mut> UpperBound() {
        return Ugram_.UpperBound(Index_);
    }

    TBucketValueRef<mut> Value() {
        return Ugram_.Value(Index_);
    };

    TIteratorBase& operator++() {
        ++Index_;
        return *this;
    }

    TIteratorBase operator+(size_t d) const {
        return TIteratorBase(Ugram_, Index_ + d);
    }

    bool operator==(TIteratorBase other) const {
        return other.Index_ == Index_;
    }

    bool operator!=(TIteratorBase other) const {
        return !(*this == other);
    }

    TUgramLikeBucket operator*() const {
        return TUgramLikeBucket{Ugram_.LowerBound(Index_), Ugram_.UpperBound(Index_), Ugram_.Value(Index_)};
    }

    TIteratorBase(const TIteratorBase<true>& it)
        : Ugram_(it.Ugram_)
        , Index_(it.Index_)
    {
    }

    TIteratorBase& operator=(const TIteratorBase<true>& other) {
        Ugram_ = other.Ugram_;
        Index_ = other.Index_;
    }

private:
    TIteratorBase(TUgramLikeWrapperBase<mut> ugram, size_t index)
        : Ugram_(ugram)
        , Index_(index)
    {
    }

private:
    TUgramLikeWrapperBase<mut> Ugram_;
    size_t Index_;
};

template <bool mut>
TUgramLikeWrapper::TIteratorBase<mut> TUgramLikeWrapperBase<mut>::begin() {
    return TIteratorBase<mut>(*this, 0u);
}

template <bool mut>
TUgramLikeWrapper::TIteratorBase<mut> TUgramLikeWrapperBase<mut>::end() {
    return TIteratorBase<mut>(*this, Count());
}

template <bool mut>
TUgramLikeWrapper::TConstIterator TUgramLikeWrapperBase<mut>::begin() const {
    return TUgramLikeWrapper::TConstIterator(*this, 0u);
}

template <bool mut>
TUgramLikeWrapper::TConstIterator TUgramLikeWrapperBase<mut>::end() const {
    return TConstIterator(*this, Count());
}

using TMutableUgramLikeWrapper = TUgramLikeWrapperBase<true>;
using TConstUgramLikeWrapper = TUgramLikeWrapperBase<false>;

static bool DifferentBounds(const TBuckets& lhs, const TBuckets& rhs) {
    if (lhs.size() != rhs.size()) {
        return true;
    }

    for (size_t i = 0; i < lhs.size(); ++i) {
        if (lhs[i].UpperBound != rhs[i].UpperBound) {
            return true;
        }
    }

    return false;
}

static void NormalizeBuckets(TBuckets& buckets) {
    if (buckets.empty() || buckets.back().UpperBound != NMonitoring::HISTOGRAM_INF_BOUND) {
        buckets.emplace_back(TBucket{NMonitoring::HISTOGRAM_INF_BOUND, 0u});
    }
}

static bool HaveSameBounds(TUgramLikeWrapper::TConstIterator lhs, TUgramLikeWrapper::TConstIterator rhs) {
    return lhs.UpperBound() == rhs.UpperBound() && lhs.LowerBound() == rhs.LowerBound();
}

static bool IsPoint(TUgramLikeWrapper::TConstIterator it) {
    return it.LowerBound() == it.UpperBound();
}

static void Spread(TUgramLikeWrapper::TConstIterator from, TUgramLikeWrapper::TIterator to) {
    if (IsPoint(to)) {
        return;
    } else {
        double overlapPart;
        if (IsPoint(from)) {
            overlapPart = 1;
        } else {
            double overlapSize = (
                    Min(from.UpperBound(), to.UpperBound()) - Max(from.LowerBound(), to.LowerBound())
            );
            overlapPart = overlapSize / (from.UpperBound() - from.LowerBound());
        }

        to.Value() += round(from.Value() * overlapPart);
    }
}

static TBuckets RemoveZerosCompress(const TBuckets& buckets) {
    if (buckets.empty()) {
        return {};
    }

    TBuckets r;
    r.emplace_back(buckets[0]);

    for (size_t i = 1u; i < buckets.size(); ++i) {
        if (buckets[i].Value == 0u) {
            continue;
        }

        r.emplace_back(buckets[i]);
    }

    NormalizeBuckets(r);

    if (!BucketsIsDefined(r)) {
        return TBuckets{{NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
    }

    return r;
}

static TBuckets GolovanCompress(const TBuckets& buckets, size_t limit) {
    NZoom::NHgram::TUgramBuckets yasmBuckets;

    auto wrapper = TConstUgramLikeWrapper(buckets);
    for (auto bucket: wrapper) {
        yasmBuckets.push_back({bucket.LowerBound, bucket.UpperBound, static_cast<double>(bucket.Value)});
    }

    auto yasmCompressed = NZoom::NHgram::TUgramCompressor::GetInstance().Compress(yasmBuckets, limit);

    TBuckets r;
    TUgramLikeInserter inserter(r);
    for (auto bucket: yasmCompressed) {
        inserter.AddBucket(bucket.LowerBound, bucket.LowerBound, bucket.Weight);
    }

    NormalizeBuckets(r);

    if (!BucketsIsDefined(r)) {
        return TBuckets{{NMonitoring::HISTOGRAM_INF_BOUND, 0u}};
    }

    return r;
}

static bool AlignerCompress(TBuckets& buckets, size_t limit) {
    auto compressed = RemoveZerosCompress(buckets);
    if (compressed.size() > limit) {
        compressed = GolovanCompress(buckets, limit - 2);
    }
    if ((BucketsIsEmpty(compressed) || !BucketsIsDefined(compressed)) && !BucketsIsEmpty(buckets)) {
        if (AllOf(buckets, [](const auto& bucket) {return bucket.Value == 0;})) {
            Y_VERIFY(buckets.size() > 1u && buckets.back().UpperBound == NMonitoring::HISTOGRAM_INF_BOUND);

            compressed = TBuckets{
                {buckets[0].UpperBound, 0u},
                {(buckets.rbegin() + 1u)->UpperBound, 0u},
                {NMonitoring::HISTOGRAM_INF_BOUND, 0u}
            };
        }
    }
    bool wasCompressed = DifferentBounds(compressed, buckets);
    buckets = std::move(compressed);
    return wasCompressed;
}

void TUgramAlignerBuilder::AddBuckets(const TBuckets& buckets) {
    if (!BucketsIsDefined(buckets) || !BucketsIsNormalized(buckets)) {
        return;
    }

    if (Buckets_.empty()) {
        Buckets_ = buckets;
        NormalizeBuckets(Buckets_);
        return;
    }

    if (!DifferentBounds(Buckets_, buckets)) {
        return;
    }

    DifferentBounds_ = true;

    auto bucketsCopy = std::move(Buckets_);

    TConstUgramLikeWrapper lhs(bucketsCopy);
    TConstUgramLikeWrapper rhs(buckets);


    TUgramLikeInserter inserter(Buckets_);
    try {
        std::merge(lhs.begin(), lhs.end(), rhs.begin(), rhs.end(), std::back_inserter(inserter),
                   [](auto lhs, auto rhs) {
                       return std::make_pair(lhs.LowerBound, lhs.UpperBound) < std::make_pair(rhs.LowerBound, rhs.UpperBound);
                   });
    } catch (...) {
        ythrow yexception() << "error while merging histograms " << BucketsToStr(bucketsCopy)
                            << " and " << BucketsToStr(buckets) << ": " << CurrentExceptionMessage();
    }

    NormalizeBuckets(Buckets_);
}

TUgramAligner::TUgramAligner(TBuckets buckets)
    : Aligner_(std::move(buckets))
{
}

TBuckets TUgramAligner::Align(const TBuckets& buckets) {
    Y_ENSURE(BucketsIsNormalized(buckets) && BucketsIsDefined(buckets), "invalid histogram: " << BucketsToStr(buckets));

    TMutableUgramLikeWrapper frozenWrapper(Aligner_);
    TConstUgramLikeWrapper forFreezeWrapper(buckets);

    auto frozen = frozenWrapper.begin();
    frozen.Value() = 0;
    auto forFreeze = forFreezeWrapper.begin();

    if (forFreezeWrapper.Count() == 0u) {
        for (auto& bucket: Aligner_) {
            bucket.Value = 0u;
        }

        return Aligner_;
    }

    Y_ENSURE(frozenWrapper.Count() > 0, "frozen buckets is empty");

    bool getNextFrozenBucket{false};

    do {
        if (getNextFrozenBucket) {
            ++frozen;
            if (frozen == frozenWrapper.end()) {
                break;
            } else {
                frozen.Value() = 0;
            }
        }

        if (HaveSameBounds(frozen, forFreeze)) {
            frozen.Value() += forFreeze.Value();
            ++forFreeze;
            getNextFrozenBucket = true;
        } else {
            if (forFreeze.LowerBound() >= frozen.UpperBound()) {
                auto nextFrozen = frozen + 1u;
                bool isLastFrozenBucket = (nextFrozen == frozenWrapper.end());
                bool isPointAtTheEnd = (
                        IsPoint(forFreeze) &&
                        forFreeze.LowerBound() == frozen.UpperBound()
                );
                if (isLastFrozenBucket || (isPointAtTheEnd && !HaveSameBounds(forFreeze, nextFrozen))) {
                    frozen.Value() += forFreeze.Value();
                    ++forFreeze;
                }
                getNextFrozenBucket = true;
            } else {
                Spread(forFreeze, frozen);
                getNextFrozenBucket = forFreeze.UpperBound() >= frozen.UpperBound();
                if (forFreeze.UpperBound() <= frozen.UpperBound()) {
                    ++forFreeze;
                }
            }
        }

    } while (forFreeze != forFreezeWrapper.end());

    if (frozen != frozenWrapper.end()) {
        ++frozen;
    }
    for (; frozen != frozenWrapper.end(); ++frozen) {
        frozen.Value() = 0;
    }

    // TODO: remove after testing
    VerifyHistogram(Aligner_);

    return Aligner_;
}

void TUgramAlignerBuilder::Reset() {
    Buckets_ = {};
    DifferentBounds_ = false;
}

std::optional<TUgramAligner> TUgramAlignerBuilder::ReleaseAligner() {
    if (Buckets_.empty()) {
        return std::nullopt;
    }

    auto compressed = AlignerCompress(Buckets_, NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT);
    if (DifferentBounds_ || compressed) {
        return std::move(Buckets_);
    }

    return std::nullopt;
}

bool Compress(TBuckets& buckets, size_t limit) {
    if (buckets.size() < limit) {
        return false;
    }

    buckets = RemoveZerosCompress(buckets);
    if (buckets.size() > limit) {
        buckets = GolovanCompress(buckets, limit - 2);
    }

    return true;
}

void TUgramBuilder::AddBucket(TBucketBound lower, TBucketBound upper, TBucketValue value) {
    TUgramLikeInserter inserter(Buckets_);
    inserter.AddBucket(lower, upper, value);
}

void TUgramBuilder::AddBucket(TBucketBound upper, TBucketValue value) {
    if (Buckets_.empty()) {
        Buckets_.emplace_back(TBucket{upper, value});
    } else {
        if (Buckets_.back().UpperBound == upper) {
            TUgramLikeInserter inserter(Buckets_);
            inserter.AddBucket(upper, upper, value);
        } else if (Buckets_.back().UpperBound < upper) {
            Buckets_.emplace_back(TBucket{upper, value});
        } else {
            ythrow yexception() << "histogram bounds must be sorted";
        }
    }
}

TBuckets TUgramBuilder::ReleaseBuckets() {
    NormalizeBuckets(Buckets_);
    Compress(Buckets_);

    // TODO: remove after testing
    VerifyHistogram(Buckets_);

    return std::move(Buckets_);
}

} // NSolomon::NMemStore::NYasm
