#include "increasing_request.h"
#include "index_benchmark.h"

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/slog/testlib/testlib.h>

#include <solomon/services/memstore/config/memstore_benchmark_config.pb.h>

namespace NSolomon::NMemStore::NBenchmark {

using yandex::monitoring::memstore::MemStoreBenchmarkConfig_ReadType_ALL;
using yandex::monitoring::memstore::MemStoreBenchmarkConfig_ReadType_ONE_METRIC;
using yandex::monitoring::memstore::MemStoreBenchmarkConfig_MetricGeneration_REPEATED_REQUEST;

TRpcRequest::TRpcRequest(MemStoreBenchmarkConfig config)
        : Config(std::move(config))
        , NumOfHosts(Config.write_request_weight())
        , NumOfHostsInc(Config.write_request_weight_increase())
        , Delay(FromProtoTime(Config.delay()))
        , ReadCount(Config.read_request_count())
{
}

auto TRpcRequest::GetMetaData(std::string prefix) const {
    auto log = NSolomon::NSlog::MakeSlog(TIndexBenchmark::shardNumId);

    if (Config.metric_generation_type() == MemStoreBenchmarkConfig_MetricGeneration_REPEATED_REQUEST) {
        prefix.clear();
    }

    for (ui32 hostId = 0; hostId < NumOfHosts; ++hostId)
        log.Gauge({{"host", "xx" + prefix + "_" + std::to_string(hostId) + "_"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
                .Done();
    return log.Done();
};

ui64 TRpcRequest::CalcRpc(TIndexBenchmark& benchmark) {
    auto snapshotSize = benchmark.Config.ChunkLength.MilliSeconds() /
            Delay.MilliSeconds() * Config.concurrency() * NumOfHosts;

    auto& actorSystem = benchmark.ActorSystem();

    auto indexCollector = actorSystem.Register(CreateIndexDoneCollector(Max<ui32>()).release());
    auto snapshotCollector = actorSystem.Register(CreateSnapshotCollector(Max<ui32>(), snapshotSize).release());
    auto readCollector = actorSystem.Register(CreateReadResponseCollector(snapshotSize).release());

    benchmark.SetSnapshotCollector(snapshotCollector);
    auto queueSizeFuture = benchmark.SetQueueSizeCollector(Config.max_queue_size());

    Cout << "Start at:" << actorSystem.Timestamp() << Endl;

    Sleep(TDuration::Seconds(3));

    for (ui32 idx = 0; !queueSizeFuture.HasValue(); ++idx) {
        for (ui32 requestId = 0; requestId < Config.concurrency(); ++requestId) {
            auto prefix = std::to_string(requestId) + "_" + std::to_string(idx);
            auto[meta, data] = GetMetaData(prefix);
            TShardKey ShardKey{"project", "cluster", "service"};
            actorSystem.Send(benchmark.ShardId, new TShardEvents::TIndex{
                {10, benchmark.GlobalLogId()},
                {TIndexBenchmark::shardNumId, ShardKey, {
                    {std::move(meta), std::move(data), indexCollector, 0}
                }}
            });

            for (ui32 readId = 0; readId < ReadCount; readId++) {
                auto* req = new TShardEvents::TReadMany;
                req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
                if (Config.read_type() == MemStoreBenchmarkConfig_ReadType_ALL) {
                    req->Lookup->Selectors.Add("host", "xx*");
                } else if (Config.read_type() == MemStoreBenchmarkConfig_ReadType_ONE_METRIC) {
                    req->Lookup->Selectors.Add("host", ToString(requestId) + "_" + ToString(idx));
                }

                actorSystem.Send(new NActors::IEventHandle{benchmark.ShardId, readCollector, req});
            }
        }
        Sleep(Delay);

        auto k = static_cast<ui32>(Config.increase_delay() * benchmark.Config.IndexCapacity.SecondsFloat() / Delay.SecondsFloat());
        if (idx > 0 && idx % k == 0) {
            NumOfHosts += NumOfHostsInc;
            ReadCount += Config.read_request_count_increase();
            Cout << "Current RPC = " << CurrentWriteRpc() << " / " << CurrentReadRpc() << Endl;
        }
    }

    Cout << "Current RPC = " << CurrentWriteRpc() << " / " << CurrentReadRpc() << Endl;

    return CurrentWriteRpc();
}

ui32 TRpcRequest::CurrentWriteRpc() const {
    return 1.f / Delay.SecondsFloat() * Config.concurrency() * NumOfHosts;
}

ui32 TRpcRequest::CurrentReadRpc() const {
    return 1.f / Delay.SecondsFloat() * Config.concurrency() * ReadCount;
}

} // namespace NSolomon::NMemStore::NBenchmark
