#include "index_result_collector.h"

#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/wal/wal_events.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/slog/testlib/testlib.h>

using namespace NActors;

namespace NSolomon::NMemStore::NBenchmark {
namespace {

/**
 * Used only in benchmark
 */
class TIndexDoneCollector: public TActor<TIndexDoneCollector> {
#define LOG_PI "{IndexDoneCollector " << SelfId() << "} "
public:
    TIndexDoneCollector(size_t expectedNumRequest, NThreading::TPromise<void> promise)
        : TActor<TIndexDoneCollector>{&TThis::StateFunc}
        , ExpectedNumRequest_(expectedNumRequest)
        , Promise_(std::move(promise))
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardEvents::TIndexDone, Handle)
        }
    }

    void Handle(const NIndex::TShardEvents::TIndexDone::TPtr&) {
        Y_ASSERT(ExpectedNumRequest_ > 0);
        ExpectedNumRequest_--;
        MON_DEBUG(Index, LOG_PI << "Index done, requests left: " << ExpectedNumRequest_);
        if (ExpectedNumRequest_ == 0 && Promise_.Initialized()) {
            Promise_.SetValue();
            MON_INFO(Index, LOG_PI << "Index Promise done.");
        }
    }

    NThreading::TFuture<void> Future() const {
        return Promise_.GetFuture();
    }

private:
    size_t ExpectedNumRequest_;
    NThreading::TPromise<void> Promise_;
};

class TSnapshotCollector: public TActor<TSnapshotCollector> {
#define LOG_PS "{SnapshotCollector " << SelfId() << "} "
public:
    TSnapshotCollector(size_t expectedNumRequest, size_t snapshotSize, NThreading::TPromise<void> promise)
        : TActor<TSnapshotCollector>{&TThis::StateFunc}
        , ExpectedNumRequest_(expectedNumRequest)
        , SnapshotSize_(snapshotSize)
        , Promise_(std::move(promise))
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NWal::TWalEvents::TSnapshot, Handle)
        }
    }

    void Handle(const NWal::TWalEvents::TSnapshot::TPtr& handle) {
        Y_ASSERT(ExpectedNumRequest_ > 0);
        ExpectedNumRequest_--;
        MON_DEBUG(Index, LOG_PS << "Snapshot done, requests left: " << ExpectedNumRequest_);

        auto ev = handle->Get();

        if (SnapshotSize_ > 0) {
            auto it = NSolomon::NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(ev->Meta);
            // TODO: add strong check
            Y_VERIFY(it->TotalMetricCount() > 0, "snapshot is not right");

        }

        if (ExpectedNumRequest_ == 0 && Promise_.Initialized()) {
            Promise_.SetValue();
            MON_DEBUG(Index, LOG_PS << "Snapshot Promise done.");
        }
    }

    NThreading::TFuture<void> Future() const {
        return Promise_.GetFuture();
    }

private:
    size_t ExpectedNumRequest_;
    size_t SnapshotSize_;
    NThreading::TPromise<void> Promise_;
};

class TReadResponseCollector: public TActor<TReadResponseCollector> {
#define LOG_PR "{ReadResponseCollector " << SelfId() << "} "
public:
    TReadResponseCollector(size_t expectedNumMetrics, NThreading::TPromise<void> promise)
        : TActor<TReadResponseCollector>{&TThis::StateFunc}
        , ExpectedNumMetrics_(expectedNumMetrics)
        , Promise_(std::move(promise))
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardEvents::TReadManyResponse, OnReadManyResponse)
            hFunc(NIndex::TShardEvents::TReadError, OnError)
        }
    }

    void OnReadManyResponse(NIndex::TShardEvents::TReadManyResponse::TPtr&) {
        Y_ENSURE(ExpectedNumMetrics_);
        // TODO: check readed data correctly
//        Y_ASSERT(!ev->Value().metrics().empty());
        Promise_.SetValue();
    }

    void OnError(NIndex::TShardEvents::TReadError::TPtr& ev) {
        Promise_.SetException(ev->Get()->Message);
    }

    NThreading::TFuture<void> Future() const {
        return Promise_.GetFuture();
    }

private:
    size_t ExpectedNumMetrics_;
    NThreading::TPromise<void> Promise_;
};

} // namespace

std::unique_ptr<NActors::IActor> CreateIndexDoneCollector(
        size_t expectedNumRequest,
        NThreading::TPromise<void> promise)
{
    return std::make_unique<TIndexDoneCollector>(expectedNumRequest, std::move(promise));
}

std::unique_ptr<NActors::IActor> CreateSnapshotCollector(
        size_t expectedNumRequest,
        size_t snapshotSize,
        NThreading::TPromise<void> promise)
{
    return std::make_unique<TSnapshotCollector>(expectedNumRequest, snapshotSize, std::move(promise));
}

std::unique_ptr<NActors::IActor> CreateReadResponseCollector(
        size_t expectedNumMetrics,
        NThreading::TPromise<void> promise)
{
    return std::make_unique<TReadResponseCollector>(expectedNumMetrics, std::move(promise));
}

} // namespace NSolomon::NMemStore::NBenchmark
