#pragma once

#include "metric_meta.h"

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/threading/future/fwd.h>

#include <util/generic/vector.h>

namespace NYdb::NTable {
    class TTableClient;
} // namespace NYdb::NTable

namespace NMetabase {

// shortcuts
using TAsyncUi64 = NThreading::TFuture<ui64>;
using TAsyncVoid = NThreading::TFuture<void>;
using TAsyncMetrics = NThreading::TFuture<TVector<TMetricMeta>>;

/**
 * Data access object to manipulate metrics persisted in backed storage.
 */
class IMetricsDao: public TThrRefBase {
public:

    /**
     * Create table in backed database to store metrics.
     *
     * @return future which will be completed when table created,
     *         or a non retryable error occured.
     */
    virtual TAsyncVoid CreateTable() = 0;

    /**
     * Count number of metrics stored in database.
     *
     * @return future with metric count or a non retryable error.
     */
    virtual TAsyncUi64 Count() const = 0;

    /**
     * Loads all metrics from database.
     *
     * @return future which will be complemeted when all metrics are loaded,
     *         or a non retryable error occured.
     */
    virtual TAsyncMetrics ReadAll() const = 0;

    /**
     * Blindly inserts given metrics into database. New metrics will be created
     * present metrics will be overriden.
     *
     * @return future which will be completed when all metrics are replaced,
     *         or a non retryable error occured.
     */
    virtual TAsyncVoid Replace(const TVector<TMetricMeta>& metrics) = 0;

    /**
     * Deletes metrics by given labels from database.
     *
     * @return future which will be completed when all metrics deleted replaced,
     *         or a non retryable error occured.
     */
    virtual TAsyncVoid Delete(const TVector<TString>& labels) = 0;
};

using IMetricsDaoPtr = TIntrusivePtr<IMetricsDao>;

/**
 * Creates implementation of non persistable storage.
 */
IMetricsDaoPtr InMemoryMetricsDao();

/**
 * Creates implementation to access metrics stored in YDB.
 */
IMetricsDaoPtr YdbMetricsDao(
    ui32 shardId,
    TString path,
    std::shared_ptr<NYdb::NTable::TTableClient> client,
    NMonitoring::TMetricRegistry& registry);

} // namespace NMetabase
