#include <solomon/services/metabase/lib/db/metrics_dao.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/algorithm.h>
#include <util/stream/file.h>
#include <util/string/builder.h>

namespace NMetabase {

static NMonitoring::TMetricRegistry Metrics;

template <typename TDaoFactory>
class TMetricsDaoTest: public ::testing::Test {
protected:
    IMetricsDaoPtr CreateDao(TStringBuf testName, ui32 shardId) {
        return TDaoFactory::Create(testName, shardId);
    }

    TVector<TMetricMeta> ReadSorted(const IMetricsDao& dao) {
        auto metrics = dao.ReadAll().ExtractValueSync();
        SortBy(metrics, [](auto& m) { return m.LocalId; });
        return metrics;
    }
};

struct TInMemory {
    static IMetricsDaoPtr Create(TStringBuf /*testName*/, ui32 /*shardId*/) {
        return InMemoryMetricsDao();
    }
};

struct TYdb {
    static IMetricsDaoPtr Create(TStringBuf testName, ui32 shardId) {
        TString endpoint = TFileInput("ydb_endpoint.txt").ReadLine();
        TString database = TFileInput("ydb_database.txt").ReadLine();

        NYdb::TDriver driver(NYdb::TDriverConfig()
                .SetEndpoint(endpoint)
                .SetDatabase(database));

        auto tableClient = std::make_shared<NYdb::NTable::TTableClient>(driver);

        auto tablePath = TStringBuilder()
                << '/' << database << '/' << getpid() << '/' << testName;

        auto dao = YdbMetricsDao(shardId, std::move(tablePath), tableClient, Metrics);
        dao->CreateTable().GetValueSync();
        return dao;
    }
};

using TMetricsDaoTypes = ::testing::Types<TInMemory, TYdb>;
TYPED_TEST_SUITE(TMetricsDaoTest, TMetricsDaoTypes);

TYPED_TEST(TMetricsDaoTest, Count) {
    IMetricsDaoPtr dao = this->CreateDao("Count", 42);
    TInstant now = TInstant::ParseIso8601("2019-10-04T01:02:03Z");

    // (1) empty database
    {
        ui64 count = dao->Count().ExtractValueSync();
        ASSERT_EQ(count, 0u);
    }

    // (2) non empty database
    {
        TMetricMeta m1{"&host=solomon-test-sas-00&name=cpu_usage&", 1, 1, 0, now};
        TMetricMeta m2{"&host=solomon-test-sas-01&name=cpu_usage&", 2, 1, 0, now};
        dao->Replace({ m1, m2 }).GetValueSync();

        ui64 count = dao->Count().ExtractValueSync();
        ASSERT_EQ(count, 2u);
    }
}

TYPED_TEST(TMetricsDaoTest, ReadAll) {
    IMetricsDaoPtr dao = this->CreateDao("LoadAll", 42);
    TInstant now = TInstant::ParseIso8601("2019-10-04T01:02:03Z");

    // (1) empty database
    {
        auto metrics = dao->ReadAll().ExtractValueSync();
        ASSERT_EQ(metrics.size(), 0u);
    }

    // (2) non empty database
    {
        TMetricMeta m1{"&host=solomon-test-sas-00&name=cpu_usage&", 1, 2, 0, now};
        TMetricMeta m2{"&host=solomon-test-sas-01&name=cpu_usage&", 2, 2, 0, now};
        dao->Replace({ m1, m2 }).GetValueSync();

        auto metrics = dao->ReadAll().ExtractValueSync();
        ASSERT_EQ(metrics.size(), 2u);

        SortBy(metrics, [](auto& m) { return m.LocalId; });
        ASSERT_EQ(m1, metrics[0]);
        ASSERT_EQ(m2, metrics[1]);
    }
}

TYPED_TEST(TMetricsDaoTest, Replace) {
    IMetricsDaoPtr dao = this->CreateDao("Replace", 42);
    TInstant now = TInstant::ParseIso8601("2019-10-04T01:02:03Z");

    // (1) replace with an empty array
    {
        dao->Replace({}).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 0u);
    }

    // (2) add new metric
    {
        TMetricMeta m{"&host=solomon-test-sas-00&name=cpu_usage&", 1, 2, 0, now};
        dao->Replace({ m }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 1u);
        ASSERT_EQ(m, metrics[0]);
    }

    // (3) replace existing
    {
        TMetricMeta m{"&host=solomon-test-sas-00&name=cpu_usage&", 111, 222, 333, now + TDuration::Seconds(444)};
        dao->Replace({ m }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 1u);
        ASSERT_EQ(m, metrics[0]);
    }
}

TYPED_TEST(TMetricsDaoTest, Delete) {
    IMetricsDaoPtr dao = this->CreateDao("Delete", 42);
    TInstant now = TInstant::ParseIso8601("2019-10-04T01:02:03Z");

    TMetricMeta m1{"&host=solomon-test-sas-00&name=cpu_usage&", 1, 3, 0, now};
    TMetricMeta m2{"&host=solomon-test-sas-01&name=cpu_usage&", 2, 3, 0, now};
    TMetricMeta m3{"&host=solomon-test-sas-02&name=cpu_usage&", 3, 3, 0, now};
    TMetricMeta m4{"&host=solomon-test-sas-03&name=cpu_usage&", 4, 3, 0, now};
    TMetricMeta m5{"&host=solomon-test-sas-04&name=cpu_usage&", 5, 3, 0, now};
    dao->Replace({ m1, m2, m3, m4 }).GetValueSync();

    // (1) delete many metrics
    {
        dao->Delete({ m1.Labels, m3.Labels, m5.Labels }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 2u);
        ASSERT_EQ(m2, metrics[0]);
        ASSERT_EQ(m4, metrics[1]);
    }

    // (2) delete single metric
    {
        dao->Delete({ m2.Labels }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 1u);
        ASSERT_EQ(m4, metrics[0]);
    }

    // (3) delete missing metric
    {
        dao->Delete({ TString{"blah-blah-blah"} }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 1u);
        ASSERT_EQ(m4, metrics[0]);
    }

    // (4) delete last metric
    {
        dao->Delete({ m4.Labels }).GetValueSync();
        auto metrics = this->ReadSorted(*dao);
        ASSERT_EQ(metrics.size(), 0u);
    }
}

TYPED_TEST(TMetricsDaoTest, NonIntersecting) {
    IMetricsDaoPtr shard1 = this->CreateDao("NonIntersecting", 1);
    IMetricsDaoPtr shard2 = this->CreateDao("NonIntersecting", 2);
    TInstant now = TInstant::ParseIso8601("2019-10-04T01:02:03Z");

    TMetricMeta m1{"&host=solomon-test-sas-00&name=cpu_usage&", 1, 2, 0, now};
    TMetricMeta m2{"&host=solomon-test-sas-01&name=cpu_usage&", 2, 2, 0, now};
    TMetricMeta m3{"&host=solomon-test-sas-02&name=cpu_usage&", 3, 2, 0, now};

    shard1->Replace({ m1, m2 }).GetValueSync();
    shard2->Replace({ m3 }).GetValueSync();

    {
        ui64 count1 = shard1->Count().GetValueSync();
        ASSERT_EQ(count1, 2u);

        auto metrics1 = this->ReadSorted(*shard1);
        ASSERT_EQ(metrics1.size(), 2u);
        ASSERT_EQ(m1, metrics1[0]);
        ASSERT_EQ(m2, metrics1[1]);

        ui64 count2 = shard2->Count().GetValueSync();
        ASSERT_EQ(count2, 1u);

        auto metrics2 = this->ReadSorted(*shard2);
        ASSERT_EQ(metrics2.size(), 1u);
        ASSERT_EQ(m3, metrics2[0]);
    }

    shard1->Delete({ m2.Labels }).GetValueSync();
    shard2->Delete({ m3.Labels }).GetValueSync();

    {
        ui64 count1 = shard1->Count().GetValueSync();
        ASSERT_EQ(count1, 1u);

        auto metrics1 = this->ReadSorted(*shard1);
        ASSERT_EQ(metrics1.size(), 1u);
        ASSERT_EQ(m1, metrics1[0]);

        ui64 count2 = shard2->Count().GetValueSync();
        ASSERT_EQ(count2, 0u);

        auto metrics2 = this->ReadSorted(*shard2);
        ASSERT_EQ(metrics2.size(), 0u);
    }
}

} // namespace NMetabase
