#include "metrics_dao.h"

#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/util.h>
#include <solomon/services/metabase/lib/java_str/hash.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <utility>

using namespace NYdb;
using namespace NYdb::NTable;
using namespace NMonitoring;
using namespace NSolomon::NDb;

namespace NMetabase {
namespace {

/**
 * Queries ids.
 */
const TString METRICS_TABLE = "metrics_table";
const TString METRICS_COUNT = "metrics_count";
const TString METRICS_DELETE = "metrics_delete";
const TString METRICS_REPLACE = "metrics_replace";

/**
 * Known table columns.
 */
const TString COLUMN_LABELS = "labels";
const TString COLUMN_SP_LOCAL_ID = "spLocalId";
const TString COLUMN_SP_SHARD_ID = "spShardId";
const TString COLUMN_FLAGS = "flags";
const TString COLUMN_CREATED_AT = "createdAt";

class TMetricMetaMapper {
public:
    TMetricMetaMapper(const TResultSet& resultSet)
        : Parser_(resultSet)
        , RowCount_(resultSet.RowsCount())
        , LabelsIdx_(ColumnIndex(COLUMN_LABELS))
        , SpLocalIdIdx_(ColumnIndex(COLUMN_SP_LOCAL_ID))
        , SpShardIdIdx_(ColumnIndex(COLUMN_SP_SHARD_ID))
        , FlagsIdx_(ColumnIndex(COLUMN_FLAGS))
        , CreatedAtIdx_(ColumnIndex(COLUMN_CREATED_AT))
    {
    }

    TVector<TMetricMeta> Map() {
        TVector<TMetricMeta> metrics;
        metrics.reserve(RowCount_);

        while (Parser_.TryNextRow()) {
            TMetricMeta& m = metrics.emplace_back();
            m.Labels = *Parser_.ColumnParser(LabelsIdx_).GetOptionalUtf8();
            m.LocalId = *Parser_.ColumnParser(SpLocalIdIdx_).GetOptionalUint64();
            m.ShardId = *Parser_.ColumnParser(SpShardIdIdx_).GetOptionalUint32();
            m.Flags = *Parser_.ColumnParser(FlagsIdx_).GetOptionalUint32();
            m.CreatedAt = TInstant::Seconds(*Parser_.ColumnParser(CreatedAtIdx_).GetOptionalUint64());
        }

        return metrics;
    }

private:
    size_t ColumnIndex(const TString& name) {
        ssize_t idx = Parser_.ColumnIndex(name);
        Y_ENSURE(idx >= 0, "cannot find column: " << name);
        return static_cast<size_t>(idx);
    }

private:
    TResultSetParser Parser_;
    const size_t RowCount_;
    const size_t LabelsIdx_;
    const size_t SpLocalIdIdx_;
    const size_t SpShardIdIdx_;
    const size_t FlagsIdx_;
    const size_t CreatedAtIdx_;
};

TReadTableSettings MakeShardRange(ui32 shardId) {
    TValue from = TValueBuilder()
        .BeginTuple()
            .AddElement().OptionalUint32(shardId)
            .AddElement().EmptyOptional(EPrimitiveType::Uint32)
            .AddElement().EmptyOptional(EPrimitiveType::Utf8)
        .EndTuple()
        .Build();

    TValue to = TValueBuilder()
        .BeginTuple()
            .AddElement().OptionalUint32(shardId)
            .AddElement().OptionalUint32(Max<ui32>())
            .AddElement().OptionalUtf8("~")  // ord('~') == 126 (greatest printable ASCII char)
        .EndTuple()
        .Build();

    return TReadTableSettings()
        .From(TKeyBound::Inclusive(from))
        .To(TKeyBound::Inclusive(to));
}

class TYdbMetricsDao: public TDaoBase, public IMetricsDao {
public:
    TYdbMetricsDao(ui32 shardId, TString path, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase(std::move(client), registry)
        , TablePath_(std::move(path))
        , ShardRange_(MakeShardRange(shardId))
        , ShardId_(shardId)
    {
        Init();
    }

    TAsyncVoid CreateTable() override {
        const auto query = RawQueries_.at(METRICS_TABLE);
        return Execute<TAsyncStatus>(*Client_, [query] (TSession session) {
                return session.ExecuteSchemeQuery(query);
        }).Apply(CompleteStatus(CounterFor("createTable")));
    }

    TAsyncUi64 Count() const override {
        auto q = RawQueries_.at(METRICS_COUNT);
        auto params = TParamsBuilder()
                .AddParam("$shardId").Uint32(ShardId_).Build()
                .Build();

        auto p = NThreading::NewPromise<ui64>();
        ReadDataWithRetries(*Client_, q, std::move(params), p, [](const TDataQueryResult& r) {
            auto parser = r.GetResultSetParser(0);
            return parser.TryNextRow() ? parser.ColumnParser(0).GetUint64() : 0UL;
        }, CounterFor("count"));
        return p.GetFuture();
    }

    TAsyncMetrics ReadAll() const override {
        auto mapper = [](const TResultSet& rs) {
            TMetricMetaMapper mapper(rs);
            return mapper.Map();
        };

        auto reader = TAsyncPagedReader<TMetricMeta, decltype(mapper)>::Create(*Client_, TablePath_, mapper, CounterFor("loadAll"));
        return reader->Read(ShardRange_);
    }

    TAsyncVoid Replace(const TVector<TMetricMeta>& metrics) override {
        if (metrics.empty()) {
            return NThreading::MakeFuture();
        }

        TParamsBuilder paramsBuilder;
        {
            auto& rows = paramsBuilder.AddParam("$rows");
            rows.BeginList();
            for (const auto& m: metrics) {
                rows.AddListItem()
                    .BeginStruct()
                        .AddMember("shardId").Uint32(ShardId_)
                        .AddMember("hash").Uint32(JavaStrHash(m.Labels))
                        .AddMember("labels").Utf8(m.Labels)
                        .AddMember("spLocalId").Uint64(m.LocalId)
                        .AddMember("spShardId").Uint32(m.ShardId)
                        .AddMember("flags").Uint32(m.Flags)
                        .AddMember("createdAt").Uint64(m.CreatedAt.Seconds())
                    .EndStruct();
            }
            rows.EndList().Build();
        }

        auto params = paramsBuilder.Build();
        auto query = RawQueries_.at(METRICS_REPLACE);
        return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), query, params);
        }).Apply(CompleteStatus(CounterFor("replace")));
    }

    TAsyncVoid Delete(const TVector<TString>& labels) override {
        TParamsBuilder paramsBuilder;
        {
            auto& keys = paramsBuilder.AddParam("$keys");
            keys.BeginList();
            for (const auto& l: labels) {
                keys.AddListItem()
                    .BeginStruct()
                        .AddMember("shardId").Uint32(ShardId_)
                        .AddMember("hash").Uint32(JavaStrHash(l))
                        .AddMember("labels").Utf8(l)
                    .EndStruct();
            }
            keys.EndList().Build();
        }
        auto params = paramsBuilder.Build();

        auto query = RawQueries_.at(METRICS_DELETE);
        return Client_->RetryOperation<TDataQueryResult>([=, params{std::move(params)}] (NYdb::NTable::TSession s) {
                return ExecutePrepared(std::move(s), query, params);
        }).Apply(CompleteStatus(CounterFor("delete")));
    }

private:
    const TString& Name() const override {
        static const TString NAME{"MetricsDao"};
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
            TStringBuf("createTable"),
            TStringBuf("count"),
            TStringBuf("laod"),
            TStringBuf("loadAll"),
            TStringBuf("replace"),
            TStringBuf("delete"),
        };
        return METHODS;
    }

    THashMap<TStringBuf, TString> LoadQueries() override {
        auto keys = {
            METRICS_TABLE,
            METRICS_COUNT,
            METRICS_DELETE,
            METRICS_REPLACE,
        };

        THashMap<TStringBuf, TString> result;
        for (const TString& key: keys) {
            TString query = NResource::Find(key + ".yql");
            Y_VERIFY_DEBUG(!query.empty(), "Resource %s.yql is empty", key.c_str());
            SubstGlobal(query, TStringBuf("${table.path}"), TablePath_);
            result.emplace(key, query);
        }
        return result;
    }

private:
    const TString TablePath_;
    const TReadTableSettings ShardRange_;
    const ui32 ShardId_;
};

} // namespace

IMetricsDaoPtr YdbMetricsDao(
    ui32 shardId,
    TString path,
    std::shared_ptr<TTableClient> client,
    TMetricRegistry& registry)
{
    return new TYdbMetricsDao(shardId, std::move(path), std::move(client), registry);
}

} // namespace NMetabase
