#include "hash.h"

namespace NMetabase {
namespace {

constexpr ui16 UnicodeError = 0xfffd;

inline bool IsContinuation(ui8 b) {
    return (b & 0xc0) == 0x80;
}

inline ui16 FourBits(ui8 b) {
    return b & 0x0f;
}

inline ui16 FiveBits(ui8 b) {
    return b & 0x1f;
}

inline ui16 SixBits(ui8 b) {
    return b & 0x3f;
}

const ui8* NextUcs2(const ui8* ptr, ui16* value) {
    ui8 ch0 = *ptr, ch1, ch2;

    switch (ch0 >> 4) {
    case 0xc:
    case 0xd:
        // ch0       ch1
        // 110xxxxx  10xxxxxx
        if (Y_LIKELY(IsContinuation(ch1 = ptr[1]))) {
            *value = (FiveBits(ch0) << 6) | SixBits(ch1);
            return ptr + 2;
        }
        break;

    case 0xe:
        // ch0      ch1      ch2
        // 1110xxxx 10xxxxxx 10xxxxxx
        if (Y_LIKELY(IsContinuation(ch1 = ptr[1]))) {
            if (Y_LIKELY(IsContinuation(ch2 = ptr[2]))) {
                *value = (((FourBits(ch0) << 6) | SixBits(ch1)) << 6) | SixBits(ch2);
                return ptr + 3;
            }
        }
        break;
    }

    // skip ill-formed code unit sequence
    *value = UnicodeError;
    return ptr + 1;
}

} // namespace

ui32 JavaStrHash(const char* str, size_t len) {
    ui32 hash = 0;
    ui16 rune = 0;

    const ui8* p = reinterpret_cast<const ui8*>(str);
    const ui8* e = p + len;

    while (p < e) {
        if (*p < 0x80) {
            hash = 31 * hash + *p++;
        } else {
            p = NextUcs2(p, &rune);
            hash = (rune == UnicodeError) ? hash : 31 * hash + rune;
        }
    }

    return hash;
}

} // namespace NMetabase
