package ru.yandex.solomon.name.resolver;

import java.util.concurrent.CompletableFuture;

import io.grpc.stub.StreamObserver;

import ru.yandex.solomon.name.resolver.client.grpc.Proto;
import ru.yandex.solomon.name.resolver.handler.ServerStatusHandler;
import ru.yandex.solomon.name.resolver.handler.UpdateResourcesHandler;
import ru.yandex.solomon.name.resolver.protobuf.FindRequest;
import ru.yandex.solomon.name.resolver.protobuf.FindResponse;
import ru.yandex.solomon.name.resolver.protobuf.ResolveRequest;
import ru.yandex.solomon.name.resolver.protobuf.ResolveResponse;
import ru.yandex.solomon.name.resolver.protobuf.ResourceServiceGrpc;
import ru.yandex.solomon.name.resolver.protobuf.ServerStatusRequest;
import ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse;
import ru.yandex.solomon.name.resolver.protobuf.UpdateResourcesRequest;
import ru.yandex.solomon.name.resolver.protobuf.UpdateResourcesResponse;
import ru.yandex.solomon.name.resolver.sink.ShardSinkClient;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcResourceService extends ResourceServiceGrpc.ResourceServiceImplBase {
    private final ServerStatusHandler statusHandler;
    private final ResourceService resourceService;
    private final ShardSinkClient shardSinkClient;

    public GrpcResourceService(ServerStatusHandler statusHandler, ResourceService resourceService, ShardSinkClient shardSinkClient) {
        this.statusHandler = statusHandler;
        this.resourceService = resourceService;
        this.shardSinkClient = shardSinkClient;
    }

    @Override
    public void updateResources(UpdateResourcesRequest request, StreamObserver<UpdateResourcesResponse> responseObserver) {
        new UpdateResourcesHandler(resourceService).updateResources(request, responseObserver);
    }

    @Override
    public void serverStatus(ServerStatusRequest request, StreamObserver<ServerStatusResponse> responseObserver) {
        statusHandler.serverStatus(request, responseObserver);
    }

    @Override
    public void find(FindRequest request, StreamObserver<FindResponse> responseObserver) {
        asyncComplete(resourceService.find(Proto.fromProto(request)).thenApply(Proto::toProto), responseObserver);
    }

    @Override
    public void resolve(ResolveRequest request, StreamObserver<ResolveResponse> responseObserver) {
        asyncComplete(resourceService.resolve(Proto.fromProto(request)).thenApply(Proto::toProto), responseObserver);
    }

    @Override
    public void resolveShardAndUpdateResources(UpdateResourcesRequest request, StreamObserver<UpdateResourcesResponse> responseObserver) {
        asyncComplete(safeCall(() -> CompletableFuture.completedFuture(Proto.fromProto(request))
                .thenCompose(updateRequest -> shardSinkClient.resolveNode(request.getCloudId())
                        .thenCompose(node -> shardSinkClient.update(node, updateRequest.cloudId, updateRequest.resources, updateRequest.removeOther, updateRequest.serviceProviderId))
                        .thenApply(unused -> UpdateResourcesResponse.getDefaultInstance()))), responseObserver);
    }
}
