package ru.yandex.solomon.name.resolver;

import javax.annotation.WillCloseWhenClosed;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.logbroker.agent.client.Session;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.logbroker.ResourceFormat;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class IssueTrackerImpl implements IssueTracker {
    private static final Logger logger = LoggerFactory.getLogger(IssueTrackerImpl.class);

    @WillCloseWhenClosed
    private final Session session;
    private final ObjectMapper mapper;

    public IssueTrackerImpl(@WillCloseWhenClosed Session session) {
        this.session = session;
        this.mapper = new ObjectMapper();
    }

    @Override
    public void reindexLostAdd(String cloudId, Resource resource) {
        var issue = mapper.createObjectNode();
        issue.put("message", "resource not found in index before reindex");
        issue.set("resource", ResourceFormat.resourceToNode(resource));
        send(resource, "reindex_lost_add", issue);
    }

    @Override
    public void reindexLostUpdate(String cloudId, Resource resource, Resource prev) {
        var issue = mapper.createObjectNode();
        issue.put("message", "timestamp from reindex greater then timestamp from index, means that some updates was lost");
        issue.set("prev", ResourceFormat.resourceToNode(prev));
        issue.set("update", ResourceFormat.resourceToNode(resource));
        send(resource, "reindex_lost_update", issue);
    }

    @Override
    public void skipObsolete(String cloudId, Resource update, Resource prev) {
        var issue = mapper.createObjectNode();
        issue.put("message", "income resource has timestamp less or equal to timestamp in index, update will be rejected as obsolete");
        issue.set("prev", ResourceFormat.resourceToNode(prev));
        issue.set("update", ResourceFormat.resourceToNode(update));
        send(update, "skip_obsolete", issue);
    }

    @Override
    public void invalidResource(Resource resource, String cause) {
        var issue =  mapper.createObjectNode();
        issue.put("message", "failed on validation");
        issue.put("cause", cause);
        issue.set("resource", ResourceFormat.resourceToNode(resource));
        send(resource, "invalid", issue);
    }

    @Override
    public void reindexObsolete(String cloudId, Resource resource) {
        var issue = mapper.createObjectNode();
        issue.put("message", "latest update or reindex by resource was many times ago, maybe resolve already removed");
        issue.set("resource", ResourceFormat.resourceToNode(resource));
        send(resource, "reindex_obsolete", issue);
    }

    @Override
    public void close() throws Exception {
        session.close();
    }

    private void send(Resource resource, String issueType, ObjectNode issue) {
        try {
            var root = mapper.createObjectNode()
                    .put("ts", String.valueOf(InstantUtils.currentTimeSeconds()))
                    .put("service", resource.service)
                    .put("issueType", issueType)
                    .put("type", resource.type)
                    .put("cloudId", resource.cloudId)
                    .put("folderId", resource.folderId)
                    .put("resourceId", resource.resourceId)
                    .set("issue", issue);

            session.send(mapper.writeValueAsString(root));
        } catch (Throwable e) {
            logger.warn("fail to send issue {}:{}", issueType, issue, e);
        }
    }
}
