package ru.yandex.solomon.name.resolver;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.grpc.Status;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.annotation.Order;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverLocalShards implements Iterable<NameResolverShard> {
    private final ConcurrentMap<String, NameResolverShard> shards = new ConcurrentHashMap<>();
    private volatile boolean closed;

    @Nullable
    @NamedObjectFinderAnnotation
    public NameResolverShard getShardById(String shardId) {
        NameResolverShard shard = shards.get(shardId);
        if (shard != null && shard.getState() == ShardState.CLOSED) {
            if (!closed) {
                remove(shard);
            }
            return null;
        }
        return shard;
    }

    public boolean addShard(NameResolverShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return shards.putIfAbsent(shard.cloudId, shard) == null;
    }

    public boolean remove(NameResolverShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return shards.remove(shard.cloudId, shard);
    }

    public Stream<NameResolverShard> stream() {
        return shards.keySet().stream()
            .map(this::getShardById)
            .filter(Objects::nonNull);
    }

    @Override
    @Nonnull
    public Iterator<NameResolverShard> iterator() {
        return new Iterator<>() {
            private final Iterator<NameResolverShard> it = shards.values().iterator();
            private NameResolverShard next;

            @Override
            public boolean hasNext() {
                while (it.hasNext()) {
                    next = it.next();
                    if (next.getState() == ShardState.CLOSED) {
                        it.remove();
                        continue;
                    }
                    return true;
                }
                next = null;
                return false;
            }

            @Override
            public NameResolverShard next() {
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }

    @Order(1)
    @EventListener(ContextClosedEvent.class)
    public CompletableFuture<Void> gracefulShutdown() {
        closed = true;
        var futures = new ArrayList<CompletableFuture<Void>>();
        var it = shards.values().iterator();
        while (it.hasNext()) {
            var shard = it.next();
            it.remove();
            futures.add(shard.stop());
        }
        return CompletableFutures.allOfVoid(futures);
    }
}
