package ru.yandex.solomon.name.resolver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import io.grpc.Status;
import it.unimi.dsi.fastutil.objects.Object2LongMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.ResourceLabels;
import ru.yandex.solomon.name.resolver.db.ResourcesDao;
import ru.yandex.solomon.name.resolver.index.ResourcesCollection;
import ru.yandex.solomon.name.resolver.index.ResourcesCollectionImpl;
import ru.yandex.solomon.name.resolver.stats.ResourceKey;
import ru.yandex.solomon.name.resolver.stats.ResourceUpdateStats;
import ru.yandex.solomon.selfmon.executors.CpuMeasureExecutor;
import ru.yandex.solomon.util.async.InFlightLimiter;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverShard implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(NameResolverShard.class);
    private static final InFlightLimiter LOAD_LIMITER = new InFlightLimiter(900);
    private static final InFlightLimiter DAO_CHANGE_LIMITER = new InFlightLimiter(10);
    private static final int BATCH_MAX_SIZE = 1000;

    public final String cloudId;
    public final AssignmentSeqNo seqNo;
    private final NameResolverShardMetrics metrics;
    private final ResourcesDao dao;
    private final IssueTracker issueTracker;
    private final Executor executor;
    private final ActorRunner actor;
    private ResourcesCollection resources;
    private final ArrayListLockQueue<UpdateRequest> updateRequests = new ArrayListLockQueue<>();
    private final ArrayListLockQueue<DeleteRequest> deleteRequests = new ArrayListLockQueue<>();
    private final ArrayListLockQueue<Runnable> runnablesQueue = new ArrayListLockQueue<>();
    private final YdbRetry retry;
    private volatile long loadInflightId = -1;
    private final CompletableFuture<Void> ready = new CompletableFuture<>();
    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    private final AtomicReference<ShardState> state = new AtomicReference<>(ShardState.INIT);

    public Throwable lastError;

    public NameResolverShard(String cloudId, AssignmentSeqNo seqNo, ResourcesDao dao, IssueTracker issueTracker, GlobalShardMetrics metrics, Executor executor, ScheduledExecutorService timer) {
        this.cloudId = cloudId;
        this.seqNo = seqNo;
        this.metrics = new NameResolverShardMetrics(cloudId, metrics);
        this.dao = dao;
        this.issueTracker = issueTracker;
        this.executor = executor;
        this.retry = new YdbRetry(this::error, timer);
        this.actor = new ActorRunner(this::act, new CpuMeasureExecutor(this.metrics.cpuTimeNanos, executor));
    }

    public ShardState getState() {
        return state.get();
    }

    public NameResolverShardMetrics metrics() {
        return metrics;
    }

    public void updateObsolete(Object2LongMap<ResourceKey> obsolete) {
        metrics.obsolete(obsolete);
    }

    public Stream<Resource> search(Selectors selectors) {
        ensureShardReady();
        logger.info("search at {} by selectors {}", cloudId, selectors);
        for (var selector : selectors) {
            if (!ResourceLabels.RESOURCE.equals(selector.getKey())) {
                continue;
            }

            if (!selector.isExact()) {
                continue;
            }

            var resource = resources.getOrNull(selector.getValue());
            if (resource == null || !ResourceLabels.match(resource, selectors)) {
                break;
            }

            return Stream.of(resource);
        }
        return resources.search(selectors);
    }

    public List<Resource> resolve(Collection<String> resourceIds) {
        ensureShardReady();
        List<Resource> result = new ArrayList<>(resourceIds.size());
        for (var resourceId : resourceIds) {
            var resource = resources.getOrNull(resourceId);
            if (resource != null) {
                result.add(resource);
            }
        }
        return result;
    }

    public CompletableFuture<Void> update(Resource resource) {
        var request = new UpdateRequest(resource);
        updateRequests.enqueue(request);
        actor.schedule();
        return request.future;
    }

    public CompletableFuture<Void> update(List<Resource> resources, boolean removeOtherOfService, String serviceProviderId) {
        var requests = new ArrayList<UpdateRequest>(resources.size());
        var futures = new CompletableFuture[resources.size()];
        for (int index = 0; index < resources.size(); index++) {
            var req = new UpdateRequest(resources.get(index));
            requests.add(req);
            futures[index] = req.future;
        }
        updateRequests.enqueueAll(requests);
        if (removeOtherOfService) {
            var request = new DeleteRequest(getResourcesToDelete(resources, serviceProviderId));
            deleteRequests.enqueue(request);
            actor.schedule();
            return CompletableFuture.allOf(futures)
                    .thenCompose(unused -> request.future);
        } else {
            actor.schedule();
            return CompletableFuture.allOf(futures);
        }
    }

    private List<Resource> getResourcesToDelete(List<Resource> resourcesUpdated, String serviceProviderId) {
        if (serviceProviderId.isEmpty()) {
            return List.of();
        }
        var ids = resourcesUpdated.stream()
                .map(Resource::getResourceId)
                .collect(Collectors.toSet());
        List<Resource> toDelete = new ArrayList<>();
        long now = System.currentTimeMillis();
        long notAfter = now - TimeUnit.HOURS.toMillis(2);
        var iterator = resourceIterator();
        while (iterator.hasNext()) {
            var resource = iterator.next();
            if (!ids.contains(resource.resourceId)) {
                if (serviceProviderId.equals(resource.service)
                        && resource.updatedAt < notAfter
                        && resource.deletedAt == 0)
                {
                    resource.deletedAt = now;
                    resource.updatedAt = now;
                    toDelete.add(resource);
                }
            }
        }
        return toDelete;
    }

    public CompletableFuture<Void> delete(List<Resource> resources) {
        var request = new DeleteRequest(resources);
        deleteRequests.enqueue(request);
        actor.schedule();
        return request.future;
    }

    public CompletableFuture<Void> start() {
        if (!state.compareAndSet(ShardState.INIT, ShardState.LOADING)) {
            throw new IllegalStateException("Shard " + cloudId + " already started");
        }

        runInActor(() -> {
            retry.loopUntilSuccess("loadShard(" + cloudId + ")", attempt -> {
                CompletableFuture<List<Resource>> doneFuture = new CompletableFuture<>();
                loadInflightId = LOAD_LIMITER.run(() -> {
                    var future = dao.findResources(cloudId);
                    CompletableFutures.whenComplete(future, doneFuture);
                    return future;
                });
                return doneFuture;
            }).thenAccept(resources -> runInActor(() -> {
                metrics.add(ResourceUpdateStats.of(resources));
                this.resources = new ResourcesCollectionImpl(cloudId, executor, resources);
                metrics.resources.set(this.resources.size());
                setState(ShardState.RUNNING);
                ready.complete(null);
            }));
        });
        return ready;
    }

    private void ensureShardReady() {
        var state = this.state.get();
        switch (state) {
            case RUNNING:
            case UPDATING:
            case DELETING:
                return;
            default:
                throw Status.UNAVAILABLE.withDescription("shard in state " + state).asRuntimeException();
        }
    }

    public Iterator<Resource> resourceIterator() {
        ensureShardReady();
        return this.resources.iterator();
    }

    public boolean isReady() {
        var state = this.state.get();
        switch (state) {
            case RUNNING:
            case UPDATING:
            case DELETING:
                return true;
            default:
                return false;
        }
    }

    private void setState(ShardState next) {
        var prev = state.get();
        do {
            if (prev == ShardState.CLOSED) {
                return;
            }
        } while (state.compareAndSet(prev, next));
    }

    private void act() {
        processRunnables();
        processUpdates();
        processDeletes();
    }

    private void runInActor(Runnable runnable) {
        runnablesQueue.enqueue(runnable);
        actor.schedule();
    }

    private void processRunnables() {
        var runnables = runnablesQueue.dequeueAll();
        for (var runnable : runnables) {
            runnable.run();
        }
    }

    private void processUpdates() {
        switch (state.get()) {
            case CLOSED -> cancelUpdates();
            case RUNNING -> applyUpdates();
        }
    }

    private void processDeletes() {
        switch (state.get()) {
            case CLOSED -> cancelDeletes();
            case RUNNING -> applyDeletes();
        }
    }

    private void applyUpdates() {
        var requests = updateRequests.dequeueAll();
        if (requests.isEmpty()) {
            return;
        }

        var futures = new ArrayList<CompletableFuture<Void>>(requests.size());
        var resourceById = new HashMap<String, Resource>(requests.size());
        var stats = new ResourceUpdateStats(requests.size());
        for (var req : requests) {
            var resource = req.resource;
            var prev = resourceById.get(resource.resourceId);
            if (prev == null) {
                prev = resources.getOrNull(resource.resourceId);
            }

            if (prev == null) {
                if (resource.reindexAt != 0) {
                    logger.warn("{}: reindex lost add {}", cloudId, resource);
                    stats.receiveLost(resource);
                    issueTracker.reindexLostAdd(cloudId, resource);
                } else {
                    logger.info("{}: add {}", cloudId, resource);
                    stats.receiveAdd(resource);
                }
                futures.add(req.future);
                resourceById.put(resource.resourceId, resource);
                continue;
            }

            restoreTsLostByReindex(prev, resource);
            if (prev.updatedAt == resource.updatedAt && prev.reindexAt < resource.reindexAt) {
                logger.info("{}: receive reindex {}, actual {}", cloudId, resource, prev);
                stats.receiveReindex(resource);
                futures.add(req.future);
                resourceById.put(resource.resourceId, resource);
            } else if (prev.updatedAt >= resource.updatedAt) {
                logger.info("{}: skip obsolete {}, actual {}", cloudId, resource, prev);
                stats.receiveObsolete(resource);
                issueTracker.skipObsolete(cloudId, resource, prev);
                if (resourceById.containsKey(resource.resourceId)) {
                    futures.add(req.future);
                } else {
                    req.future.complete(null);
                }
            } else {
                if (resource.deletedAt != 0 && !resource.hasName()) {
                    resource.name = prev.name;
                }

                if (resource.reindexAt != 0) {
                    logger.warn("{}: reindex lost update {}, prev {}", cloudId, resource, prev);
                    stats.receiveLost(prev, resource);
                    issueTracker.reindexLostUpdate(cloudId, resource, prev);
                } else {
                    logger.info("{}: update {}, prev {}", cloudId, resource, prev);
                    stats.receiveUpdate(prev, resource);
                }

                resourceById.put(resource.resourceId, resource);
                futures.add(req.future);
            }
        }

        if (resourceById.isEmpty()) {
            metrics.add(stats);
            return;
        }

        setState(ShardState.UPDATING);
        replaceResources(resourceById.values())
                .thenRun(() -> metrics.add(stats))
                .thenCompose(ignore -> resources.putAll(resourceById.values()))
                .whenComplete((ignore, e) -> {
                    metrics.resources.set(resources.size());
                    runInActor(() -> setState(ShardState.RUNNING));
                    if (e != null) {
                        for (var future : futures) {
                            future.completeExceptionally(e);
                        }
                    } else {
                        for (var future : futures) {
                            future.complete(null);
                        }
                    }
                });
    }

    private void restoreTsLostByReindex(Resource prev, Resource resource) {
        if (resource.reindexAt == 0) {
            return;
        }

        if (resource.updatedAt != resource.reindexAt) {
            return;
        }

        if (prev.updatedAt > resource.updatedAt) {
            // obsolete reindex
            return;
        }

        if (prev.deletedAt == 0 && resource.deletedAt > 0) {
            // lost delete
            return;
        }

        if (!Objects.equals(prev.name, resource.name)
                || !Objects.equals(prev.type, resource.type)
                || !Objects.equals(prev.service, resource.service)
                || !Objects.equals(prev.folderId, resource.folderId))
        {
            // lost update
            return;
        }

        resource.updatedAt = prev.updatedAt;
        if (prev.deletedAt != 0 && resource.reindexAt == resource.deletedAt) {
            resource.deletedAt = prev.deletedAt;
        }
    }

    private void applyDeletes() {
        var requests = deleteRequests.dequeueAll();
        if (requests.isEmpty()) {
            return;
        }

        var futures = new ArrayList<CompletableFuture<Void>>(requests.size());
        List<Resource> deleteList = new ArrayList<>(requests.stream().mapToInt(r -> r.resources.size()).sum());
        var stats = new ResourceUpdateStats(requests.size());
        for (var req : requests) {
            futures.add(req.future);
            for (var resource : req.resources) {
                var prev = resources.getOrNull(resource.resourceId);
                if (prev == null || prev.updatedAt > resource.updatedAt) {
                    continue;
                }

                deleteList.add(resource);
                stats.receiveRemoveFromDb(prev);
            }
        }

        setState(ShardState.DELETING);
        deleteResources(deleteList)
                .thenRun(() -> metrics.add(stats))
                .thenCompose(ignore -> resources.removeAll(deleteList
                        .stream()
                        .map(resource -> resource.resourceId)
                        .collect(Collectors.toList())))
                .whenComplete((ignore, e) -> {
                    metrics.resources.set(resources.size());
                    runInActor(() -> setState(ShardState.RUNNING));
                    if (e != null) {
                        for (var future : futures) {
                            future.completeExceptionally(e);
                        }
                    } else {
                        for (var future : futures) {
                            future.complete(null);
                        }
                    }
                });
    }

    private CompletableFuture<Void> replaceResources(Collection<Resource> newResources) {
        return batchLimiter(newResources, batch -> {
            return retry.loopUntilSuccess("replaceResource(" + batch.size() + ")", attempt -> {
                return limitDaoInFlight(() -> dao.replaceResources(batch));
            });
        });
    }

    private CompletableFuture<Void> deleteResources(Collection<Resource> resources) {
        return batchLimiter(resources, batch -> {
            return retry.loopUntilSuccess("deleteResources(" + batch.size() + ")", attempt -> {
                return limitDaoInFlight(() -> dao.deleteResources(batch));
            });
        });
    }

    private CompletableFuture<Void> batchLimiter(Collection<Resource> resources, Function<Collection<Resource>, CompletableFuture<Void>> fn) {
        if (resources.size() <= BATCH_MAX_SIZE) {
            return fn.apply(resources);
        }

        var root = new CompletableFuture<Void>();
        var doneFuture = root;
        for (var batch : Lists.partition(new ArrayList<>(resources), BATCH_MAX_SIZE)) {
            doneFuture = doneFuture.thenCompose(ignore -> fn.apply(batch));
        }
        root.complete(null);
        return doneFuture;
    }

    private CompletableFuture<Void> limitDaoInFlight(Supplier<CompletableFuture<Void>> supplier) {
        var doneFuture = new CompletableFuture<Void>();
        DAO_CHANGE_LIMITER.run(() -> {
            if (state.get() == ShardState.CLOSED) {
                doneFuture.completeExceptionally(new CancellationException("shard already closed"));
                return doneFuture;
            }
            var future = supplier.get();
            CompletableFutures.whenComplete(future, doneFuture);
            return future;
        });
        return doneFuture;
    }

    private void cancelUpdates() {
        var requests = updateRequests.dequeueAll();
        Throwable e = Status.CANCELLED.withDescription("Shard " + cloudId + " closed").asRuntimeException();
        for (var req : requests) {
            req.future.completeExceptionally(e);
        }
        doneFuture.complete(null);
    }

    private void cancelDeletes() {
        var requests = deleteRequests.dequeueAll();
        Throwable e = Status.CANCELLED.withDescription("Shard " + cloudId + " closed").asRuntimeException();
        for (var req : requests) {
            req.future.completeExceptionally(e);
        }
        doneFuture.complete(null);
    }

    public void error(Throwable e) {
        state.compareAndSet(ShardState.LOADING, ShardState.LOADING_ERRORS);
        logger.warn("error on shard " + cloudId, e);
        lastError = e;
        metrics.error();
    }

    public CompletableFuture<Void> stop() {
        close();
        return doneFuture;
    }

    @Override
    public void close() {
        runInActor(() -> {
            var prevState = state.getAndSet(ShardState.CLOSED);
            retry.close();
            LOAD_LIMITER.remove(loadInflightId);
            if (prevState != ShardState.UPDATING && prevState != ShardState.DELETING) {
                doneFuture.complete(null);
            }
        });
    }

    @Override
    public String toString() {
        return "NameResolverShard{" +
                "cloudId='" + cloudId + '\'' +
                '}';
    }

    private static class UpdateRequest {
        private final Resource resource;
        private final CompletableFuture<Void> future = new CompletableFuture<>();

        public UpdateRequest(Resource resource) {
            this.resource = resource;
        }
    }

    private static class DeleteRequest {
        private final List<Resource> resources;
        private final CompletableFuture<Void> future = new CompletableFuture<>();

        public DeleteRequest(List<Resource> resources) {
            this.resources = resources;
        }
    }
}
