package ru.yandex.solomon.name.resolver;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.grpc.Status;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.FindResponse;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.ResolveResponse;
import ru.yandex.solomon.name.resolver.client.Resource;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceServiceImpl implements ResourceService {
    private final NameResolverLocalShards shards;

    public ResourceServiceImpl(NameResolverLocalShards shards) {
        this.shards = shards;
    }

    @Override
    public CompletableFuture<Void> updateResources(String shardId, List<Resource> resources, boolean removeOtherOfService, String serviceProviderId) {
        try {
            var shard = shards.getShardById(shardId);
            if (shard == null) {
                return CompletableFuture.failedFuture(Status.NOT_FOUND
                        .withDescription("Shard " + shardId + " absent on host")
                        .asRuntimeException());
            }

            return shard.update(resources, removeOtherOfService, serviceProviderId);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        var h = WhatThreadDoes.push("ResourceService#find " + request);
        try {
            var shard = shards.getShardById(request.cloudId);
            if (shard == null) {
                return CompletableFuture.failedFuture(Status.NOT_FOUND
                        .withDescription("Shard " + request.cloudId + " absent on host")
                        .asRuntimeException());
            }
            int skip = 0;
            int baseSkip = 0;
            if (!request.pageToken.isBlank()) {
                skip = Integer.parseInt(request.pageToken);
                baseSkip = skip;
            }

            int remaining = request.limit == 0 ? Integer.MAX_VALUE : request.limit;
            List<Resource> resources = new ArrayList<>();
            var it = shard.search(request.selectors).iterator();
            while (it.hasNext() && remaining > 0) {
                var resource = it.next();
                if (request.filterDeleted && resource.deletedAt != 0) {
                    continue;
                }
                if (request.filterReplaced && resource.replaced) {
                    continue;
                }
                if (skip > 0) {
                    skip--;
                    continue;
                }

                resources.add(resource);
                remaining--;
            }

            boolean truncated = remaining == 0 && it.hasNext();
            var nextToken = "";
            if (truncated) {
                nextToken =  (baseSkip + request.limit) + "";
            }
            return completedFuture(new FindResponse(resources, truncated, nextToken));
        } catch (Throwable e) {
            return failedFuture(e);
        } finally {
            h.popSafely();
        }
    }

    @Override
    public CompletableFuture<ResolveResponse> resolve(ResolveRequest request) {
        var h = WhatThreadDoes.push("ResourceService#resolve " + request.cloudId);
        try {
            var shard = shards.getShardById(request.cloudId);
            if (shard == null) {
                return CompletableFuture.failedFuture(Status.NOT_FOUND
                        .withDescription("Shard " + request.cloudId + " absent on host")
                        .asRuntimeException());
            }

            var resources = shard.resolve(request.resourceIds);
            return completedFuture(new ResolveResponse(resources));
        } catch (Throwable e) {
            return failedFuture(e);
        } finally {
            h.popSafely();
        }
    }
}
