package ru.yandex.solomon.name.resolver;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;

import javax.annotation.Nullable;

import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.util.actors.PingActorRunner;

import static java.util.stream.Collectors.toUnmodifiableMap;

/**
 * @author Vladimir Gordiychuk
 */
public class ServiceProviderWatcher implements AutoCloseable {
    private final ServiceProvidersDao dao;
    private final PingActorRunner actor;
    private final List<ServiceProviderListener> listeners;

    public ServiceProviderWatcher(ServiceProvidersDao dao, List<ServiceProviderListener> listeners, Executor executor, ScheduledExecutorService timer) {
        this.dao = dao;
        this.listeners = listeners;
        this.actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofMinutes(5L))
                .backoffDelay(Duration.ofMillis(128))
                .executor(executor)
                .timer(timer)
                .operation("Reload service providers")
                .onPing(value -> reloadServiceProviders())
                .build();
        actor.forcePing();
    }

    private CompletableFuture<Void> reloadServiceProviders() {
        return dao.findAll()
                .thenAccept(this::notifyListeners);
    }

    private void notifyListeners(List<ServiceProvider> serviceProviders) {
        var snapshot = serviceProviders.stream()
                .collect(toUnmodifiableMap(ServiceProvider::getId, Function.identity()));

        @Nullable
        RuntimeException error = null;
        for (var listener : listeners) {
            try {
                listener.onUpdateServiceProviders(snapshot);
            } catch (Throwable e) {
                if (error == null) {
                    error = new RuntimeException("Service provider listener " + listener + " failed", e);
                } else {
                    error.addSuppressed(e);
                }
            }
        }

        if (error != null) {
            throw error;
        }
    }

    @Override
    public void close() {
        actor.close();
    }
}
