package ru.yandex.solomon.name.resolver;

import java.util.EnumMap;
import java.util.List;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.name.resolver.stats.ByResource;
import ru.yandex.solomon.name.resolver.stats.CountDistributionMetrics;
import ru.yandex.solomon.name.resolver.stats.CountMetrics;
import ru.yandex.solomon.name.resolver.stats.ResourceKey;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardMetricSupplier implements MetricSupplier {
    private final GlobalShardMetrics globalResourceMetrics;
    private final NameResolverLocalShards shards;

    public ShardMetricSupplier(GlobalShardMetrics globalResourceMetrics, NameResolverLocalShards shards) {
        this.globalResourceMetrics = globalResourceMetrics;
        this.shards = shards;
    }

    @Override
    public int estimateCount() {
        return 0;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        var collector = new Collector();
        for (var shard : shards) {
            collector.add(shard);
            var countMetrics = shard.metrics().countMetrics;
            for (var entry : countMetrics.byKey.entrySet()) {
                var key = entry.getKey();
                var value = entry.getValue();
                value.append(tsMillis, commonLabels, consumer);
                collector.add(key, value);
            }
        }

        collector.append(tsMillis, commonLabels, consumer);
        globalResourceMetrics.append(tsMillis, commonLabels, consumer);
    }

    private static class Collector implements MetricSupplier {
        private ByResource<CountMetrics> totalCount = new ByResource<>(key -> new CountMetrics(key.toLabels()));
        private ByResource<CountDistributionMetrics> distribution = new ByResource<>(key -> new CountDistributionMetrics(key.toLabels()));
        private final MetricRegistry registry = new MetricRegistry();
        private final EnumMap<ShardState, GaugeInt64> shardsInState;

        public Collector() {
            shardsInState = EnumMetrics.gaugesInt64(ShardState.class, registry, "nameResolver.shard.count", "state");
        }

        public void add(NameResolverShard shard) {
            shardsInState.get(shard.getState()).add(1);
        }

        public void add(ResourceKey key, CountMetrics metrics) {
            for (var aggr : List.of(key.aggrType(), key.aggr())) {
                totalCount.get(aggr).combine(metrics);
                distribution.get(aggr).add(metrics);
            }
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            for (var metrics : totalCount.byKey.values()) {
                metrics.append(tsMillis, commonLabels, consumer);
            }

            for (var metrics : distribution.byKey.values()) {
                metrics.append(tsMillis, commonLabels, consumer);
            }

            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
