package ru.yandex.solomon.name.resolver;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.IntFunction;

import io.grpc.Status;

import ru.yandex.misc.concurrent.CompletableFutures;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbRetry implements AutoCloseable {
    private static final long SLEEP_MIN_MILLIS = 1_000;
    private static final long SLEEP_MAX_MILLIS = 60_000;

    private final Consumer<Throwable> exceptionConsumer;
    private final ScheduledExecutorService timer;
    public volatile boolean closed;

    public YdbRetry(Consumer<Throwable> exceptionConsumer, ScheduledExecutorService timer) {
        this.exceptionConsumer = exceptionConsumer;
        this.timer = timer;
    }

    public <T> CompletableFuture<T> loopUntilSuccess(String name, IntFunction<CompletableFuture<T>> supplier) {
        CompletableFuture<T> result = new CompletableFuture<>();
        loopUntilSuccess(name, supplier, result, SLEEP_MIN_MILLIS, 0);
        return result;
    }

    public <T> void loopUntilSuccess(
            String name,
            IntFunction<CompletableFuture<T>> supplier,
            CompletableFuture<T> resultFuture,
            long sleepMillis,
            int attempt)
    {
        if (closed) {
            resultFuture.completeExceptionally(Status.CANCELLED.withDescription("Retry context already closed").asException());
            return;
        }

        CompletableFutures.safeCall(() -> supplier.apply(attempt)).whenComplete((r, e) -> {
            try {
                if (e == null) {
                    resultFuture.complete(r);
                    return;
                }

                onError(new Exception(name + " failed, attempt " + attempt, e));
                schedule(() -> {
                    loopUntilSuccess(name, supplier, resultFuture, Math.min(sleepMillis * 2, SLEEP_MAX_MILLIS), attempt + 1);
                }, sleepMillis);
            } catch (Throwable e2) {
                resultFuture.completeExceptionally(e2);
                onError(e2);
            }
        });
    }

    private void schedule(Runnable runnable, long delayMillis) {
        long half = delayMillis / 2;
        long jitter = ThreadLocalRandom.current().nextLong(half);
        timer.schedule(runnable, half + jitter, TimeUnit.MILLISECONDS);
    }

    private void onError(Throwable e) {
        exceptionConsumer.accept(e);
    }

    @Override
    public void close() {
        closed = true;
    }
}
