package ru.yandex.solomon.name.resolver.balancer;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.name.resolver.NameResolverShardFactory;
import ru.yandex.solomon.util.file.FileStorage;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class AssignmentStore {
    private static final Logger logger = LoggerFactory.getLogger(AssignmentStore.class);
    private static final String ASSIGNMENTS_STATE_FILE = "assignments.state";

    private final FileStorage localStorage;
    private final NameResolverLocalShards shards;
    private final NameResolverShardFactory factory;
    private final ObjectMapper mapper;
    private final ActorRunner actor;
    private volatile CountDownLatch sync = new CountDownLatch(1);

    public AssignmentStore(
        FileStorage localStorage,
        NameResolverLocalShards shards,
        ExecutorService executor,
        NameResolverShardFactory factory)
    {
        this.localStorage = localStorage;
        this.shards = shards;
        this.factory = factory;
        this.mapper = new ObjectMapper();
        this.actor = new ActorRunner(this::act, executor);
        this.restoreState();
    }

    public void scheduleFlush() {
        actor.schedule();
    }

    public void awaitAct() throws InterruptedException {
        var copy = sync;
        actor.schedule();
        copy.await();
    }

    private void act() {
        saveStateToFile(shards.stream().map(Assignment::of).collect(toList()));
        var prev = sync;
        sync = new CountDownLatch(1);
        prev.countDown();
    }

    private void restoreState() {
        try {
            List<Assignment> assignments = loadStateFromFile();
            for (var assignment : assignments) {
                var seqNo = new AssignmentSeqNo(assignment.leaderSeqNo, assignment.assignSeqNo);
                var shard = factory.create(assignment.shardId, seqNo);
                if (shards.addShard(shard)) {
                    logger.info("Restore assignment {}", assignment);
                    shard.start();
                }
            }
        } catch (Throwable e) {
            logger.error("Restore assignments state failed", e);
        }
    }

    private void saveStateToFile(List<Assignment> config) {
        try {
            localStorage.save(ASSIGNMENTS_STATE_FILE, config, mapper::writeValueAsString);
        } catch (Throwable e) {
            logger.error("Error while writing state file " + ASSIGNMENTS_STATE_FILE, e);
        }
    }

    private List<Assignment> loadStateFromFile() {
        try {
            List<Assignment> assignments = localStorage.load(
                ASSIGNMENTS_STATE_FILE,
                state -> mapper.readValue(state, new TypeReference<List<Assignment>>() {}));

            return Objects.requireNonNullElseGet(assignments, List::of);
        } catch (Throwable e) {
            logger.error("Error while reading state file " + localStorage + "/" + ASSIGNMENTS_STATE_FILE, e);
            return List.of();
        }
    }

    private static class Assignment {
        public String shardId;
        public long leaderSeqNo;
        public long assignSeqNo;

        public static Assignment of(NameResolverShard shard) {
            Assignment result = new Assignment();
            result.shardId = shard.cloudId;
            result.leaderSeqNo = shard.seqNo.getLeaderSeqNo();
            result.assignSeqNo = shard.seqNo.getAssignSeqNo();
            return result;
        }

        @Override
        public String toString() {
            return "Assignment{" +
                    " shardId=" + shardId +
                    ", seqNo=" + new AssignmentSeqNo(leaderSeqNo, assignSeqNo) +
                    '}';
        }
    }
}
