package ru.yandex.solomon.name.resolver.balancer;

import java.time.Clock;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.balancer.TNodeSummary;
import ru.yandex.solomon.balancer.TShardSummary;
import ru.yandex.solomon.balancer.TShardSummary.EStatus;
import ru.yandex.solomon.balancer.protobuf.TBalancerResources;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;

/**
 * @author Vladimir Gordiychuk
 */
public class AssignmentSummary {
    private static final Logger logger = LoggerFactory.getLogger(AssignmentSummary.class);
    private final Clock clock;
    private final NameResolverLocalShards shards;
    private final long startedAt;

    public AssignmentSummary(Clock clock, NameResolverLocalShards shards) {
        this.startedAt = clock.millis();
        this.clock = clock;
        this.shards = shards;
    }

    public TNodeSummary prepareNodeSummary() {
        try {
            return TNodeSummary.newBuilder()
                    .setMemoryBytes(ProcSelfMon.getRssBytes())
                    .setUtimeMillis(ProcSelfMon.getUtimeMs())
                    .setUpTimeMillis(clock.millis() - startedAt)
                    .build();
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
            return TNodeSummary.getDefaultInstance();
        }
    }

    public List<TShardSummary> prepareShardsSummary() {
        return shards.stream()
                .map(this::prepareShardSummary)
                .collect(Collectors.toList());
    }

    private TShardSummary prepareShardSummary(NameResolverShard shard) {
        var result = TShardSummary.newBuilder();
        result.setShardId(shard.cloudId);
        switch (shard.getState()) {
            case INIT:
                result.setStatus(TShardSummary.EStatus.INIT);
                break;
            case LOADING:
            case LOADING_ERRORS:
                result.setStatus(EStatus.LOADING);
                break;
            default:
                result.setStatus(EStatus.READY);
                break;
        }

        var metrics = shard.metrics();
        result.setUpTimeMillis(metrics.uptimeMillis());
        result.setResources(TBalancerResources.newBuilder()
                .setCpu(TimeUnit.NANOSECONDS.toMillis(Math.round(metrics.cpuTimeNanos.getRate(TimeUnit.SECONDS))))
                .setResourcesCount(metrics.resources.get())
                .build());
        return result.build();
    }
}
