package ru.yandex.solomon.name.resolver.balancer;

import java.util.concurrent.CompletableFuture;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;

import ru.yandex.grpc.utils.StreamObservers;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.BalancerServiceGrpc;
import ru.yandex.solomon.balancer.TAssignShardRequest;
import ru.yandex.solomon.balancer.TAssignShardResponse;
import ru.yandex.solomon.balancer.TBalancerServiceGrpc;
import ru.yandex.solomon.balancer.TCreateShardRequest;
import ru.yandex.solomon.balancer.TCreateShardResponse;
import ru.yandex.solomon.balancer.TPingRequest;
import ru.yandex.solomon.balancer.TPingResponse;
import ru.yandex.solomon.balancer.TUnassignShardRequest;
import ru.yandex.solomon.balancer.TUnassignShardResponse;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcBalancerService extends TBalancerServiceGrpc.TBalancerServiceImplBase {
    private final AssignmentGate gate;
    private final AssignmentStore store;
    private final AssignmentSummary summary;
    private final BalancerHolder balancerHolder;

    public GrpcBalancerService(AssignmentGate gate, AssignmentStore store, AssignmentSummary summary, BalancerHolder balancerHolder) {
        this.gate = gate;
        this.store = store;
        this.summary = summary;
        this.balancerHolder = balancerHolder;
    }

    @Override
    public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
        CompletableFuture<TAssignShardResponse> future = gate.assignShard(request)
                .whenComplete((ignore, e) -> store.scheduleFlush());
        StreamObservers.asyncComplete(future, responseObserver);
    }

    @Override
    public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
        CompletableFuture<TUnassignShardResponse> future = gate.unassignShard(request)
                .whenComplete((ignore, e) -> store.scheduleFlush());
        StreamObservers.asyncComplete(future, responseObserver);
    }

    @Override
    public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
        try {
            responseObserver.onNext(TPingResponse.newBuilder()
                    .setNode(HostUtils.getFqdn())
                    .setNodeSummary(summary.prepareNodeSummary())
                    .addAllShardSummary(summary.prepareShardsSummary())
                    .build());
            responseObserver.onCompleted();
        } catch (Throwable e) {
            responseObserver.onError(e);
        }
    }

    @Override
    public void createShard(TCreateShardRequest request, StreamObserver<TCreateShardResponse> responseObserver) {
        Balancer balancer = balancerHolder.getBalancer();
        if (balancer == null) {
            responseObserver.onError(Status.FAILED_PRECONDITION
                    .withDescription("Not leader, leader is " + balancerHolder.getLeaderNode())
                    .asRuntimeException());
            return;
        }

        var future = balancer.getOrCreateAssignment(request.getShardId())
                .thenApply(node -> TCreateShardResponse.newBuilder()
                        .setNode(node)
                        .build());

        StreamObservers.asyncComplete(future, responseObserver);
    }

    public Proxy createProxy() {
        return new Proxy(this);
    }

    public static class Proxy extends BalancerServiceGrpc.BalancerServiceImplBase {
        private final GrpcBalancerService delegate;

        public Proxy(GrpcBalancerService delegate) {
            this.delegate = delegate;
        }

        @Override
        public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
            delegate.assignShard(request, responseObserver);
        }

        @Override
        public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
            delegate.unassignShard(request, responseObserver);
        }

        @Override
        public void createShard(TCreateShardRequest request, StreamObserver<TCreateShardResponse> responseObserver) {
            delegate.createShard(request, responseObserver);
        }

        @Override
        public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
            delegate.ping(request, responseObserver);
        }
    }
}
