package ru.yandex.solomon.name.resolver.balancer;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.collect.Sets;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.name.resolver.db.ShardsDao;
import ru.yandex.solomon.util.async.InFlightLimiter;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverShardsHolder implements ShardsHolder, AutoCloseable {
    private final ShardsDao dao;
    private final DistributedLock lock;
    private final ConcurrentMap<String, String> shardIds = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, CompletableFuture<Void>> deletingShards = new ConcurrentHashMap<>();
    private final InFlightLimiter deleteLimiter = new InFlightLimiter(10);
    private volatile boolean closed;

    public NameResolverShardsHolder(ShardsDao dao, DistributedLock lock) {
        this.dao = dao;
        this.lock = lock;
    }

    @Override
    public CompletableFuture<Void> reload() {
        return dao.createSchemaForTests()
                .thenCompose(ignore -> dao.findAll())
                .thenAccept(ids -> {
                    for (var id : ids) {
                        shardIds.put(id, id);
                    }
                });
    }

    @Override
    public Set<String> getShards() {
        var copy = Set.copyOf(deletingShards.keySet());
        return Sets.difference(shardIds.keySet(), copy);
    }

    @Override
    public CompletableFuture<Void> add(String shardId) {
        return CompletableFutures.safeCall(() -> {
            var deleting = deletingShards.get(shardId);
            if (deleting != null) {
                return deleting.thenCompose(ignore -> add(shardId));
            }

            if (shardIds.containsKey(shardId)) {
                return completedFuture(null);
            }

            return dao.insert(shardId).thenAccept(ignore -> shardIds.put(shardId, shardId));
        });
    }

    @Override
    public CompletableFuture<Void> delete(String shardId) {
        if (!shardIds.containsKey(shardId)) {
            return completedFuture(null);
        }

        var deleteFuture = new CompletableFuture<Void>();
        var doneFuture = deleteFuture.whenComplete((ignore, e) -> {
            if (e == null) {
                shardIds.remove(shardId);
            }
            deletingShards.remove(shardId);
        });

        var prevFuture = deletingShards.putIfAbsent(shardId, doneFuture);
        if (prevFuture != null) {
            return prevFuture;
        }

        deleteLimiter.run(() -> {
            if (!lock.isLockedByMe()) {
                deleteFuture.completeExceptionally(new IllegalStateException("Not leader anymore"));
                return deleteFuture;
            }

            if (closed) {
                deleteFuture.completeExceptionally(new IllegalStateException("Already closed"));
                return deleteFuture;
            }

            CompletableFutures.whenComplete(dao.delete(shardId), deleteFuture);
            return deleteFuture;
        });

        return doneFuture;
    }

    @Override
    public void close() {
        closed = true;
    }
}
