package ru.yandex.solomon.name.resolver.db.ydb;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.settings.AutoPartitioningPolicy;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.PartitioningPolicy;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.PrimitiveType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.db.ResourcesDao;
import ru.yandex.solomon.name.resolver.index.ResourceInternerImpl;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbResourcesDao implements ResourcesDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbResourcesDao.class);

    private static final int SELECT_LIMIT = 1000;
    private final String root;
    private final String tablePath;
    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final ObjectMapper objectMapper;

    private final String selectQuery;
    private final String replaceQuery;
    private final String deleteQuery;

    public YdbResourcesDao(String root, TableClient tableClient, SchemeClient schemeClient, ObjectMapper objectMapper) {
        this.root = root;
        this.schemeClient = schemeClient;
        this.tablePath = root + "/Resources";
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
        this.objectMapper = objectMapper;

        this.replaceQuery = String.format("""
                        --!syntax_v1
                        DECLARE $rows AS %s;
                        REPLACE INTO `%s` SELECT * FROM AS_TABLE($rows);
                        """,
                YdbResourcesTable.RESOURCES_LIST_TYPE, tablePath);

        this.deleteQuery = String.format("""
                        --!syntax_v1
                        DECLARE $keys AS %s;
                        DELETE FROM `%s` ON SELECT * FROM AS_TABLE($keys);
                        """,
                YdbResourcesTable.RESOURCES_PK_LIST_TYPE, tablePath);

        this.selectQuery = String.format("""
                        --!syntax_v1
                        DECLARE $hash AS Uint32;
                        DECLARE $cloudId AS Utf8;
                        SELECT * FROM `%s`
                        WHERE hash = $hash AND cloudId = $cloudId
                        LIMIT %s;
                        """,
                tablePath, SELECT_LIMIT);
    }

    @Override
    public CompletableFuture<Void> replaceResources(Collection<Resource> resource) {
        Params params = Params.of("$rows", YdbResourcesTable.resourcesToList(resource, objectMapper));
        return execute(replaceQuery, params)
                .thenAccept(r -> r.expect("cannot replace resources"));
    }

    @Override
    public CompletableFuture<Void> deleteResources(Collection<Resource> resources) {
        Params params = Params.of("$keys", YdbResourcesTable.resourcesToPkList(resources));
        return execute(deleteQuery, params)
                .thenAccept(r -> r.expect("cannot delete resources"));
    }

    @Override
    public CompletableFuture<List<Resource>> findResources(String cloudId) {
        return findAllBySelectTable(cloudId)
                .handle((resources, e) -> {
                    if (e != null) {
                        logger.warn("Unable find resource by select, fallback to readtable", e);
                        return findAllByReadTable(cloudId);
                    }

                    if (resources == null) {
                        // cloudId has more resources that limit for select
                        return findAllByReadTable(cloudId);
                    }

                    return completedFuture(resources);
                })
                .thenCompose(future -> future);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenCompose(ignore -> retryCtx.supplyStatus(session -> {
                    var settings = new CreateTableSettings()
                            .setPartitioningPolicy(new PartitioningPolicy().setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE))
                            .setTimeout(10, TimeUnit.SECONDS);

                    return session.createTable(tablePath, YdbResourcesTable.description(), settings);
                }))
                .thenAccept(status -> status.expect("can't create " + tablePath + " table"));
    }

    public CompletableFuture<Void> migrateSchema() {
        return retryCtx.supplyResult(session -> session.describeTable(tablePath))
                .thenCompose(result -> {
                    if (!result.isSuccess()) {
                        return createSchemaForTests();
                    }
                    return alterTable(result.expect("unable describe " + tablePath));
                });
    }

    private CompletableFuture<Void> alterTable(TableDescription table) {
        for (var column : table.getColumns()) {
            if ("payload".equals(column.getName())) {
                return completedFuture(null);
            }
        }

        var settings = new AlterTableSettings();
        settings.addColumn("payload", PrimitiveType.json().makeOptional());
        return retryCtx.supplyStatus(session -> session.alterTable(tablePath, settings))
                .thenAccept(status -> status.expect("unable alter " + tablePath));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return retryCtx.supplyStatus(session -> session.dropTable(tablePath))
                .thenAccept(status -> status.expect("can't drop " + tablePath + " table"));
    }

    private CompletableFuture<List<Resource>> findAllBySelectTable(String cloudId) {
        return execute(selectQuery, YdbResourcesTable.keyParams(cloudId))
                .thenApply(result -> {
                    DataQueryResult dataQueryResult = result.expect("can't select " + tablePath + "table");
                    ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                    if (resultSet.getRowCount() >= SELECT_LIMIT || resultSet.isTruncated()) {
                        return null;
                    }

                    YdbResourcesReader reader = new YdbResourcesReader(ResourceInternerImpl.INSTANCE, objectMapper);
                    reader.accept(resultSet);
                    return reader.resources;
                });
    }

    private CompletableFuture<List<Resource>> findAllByReadTable(String cloudId) {
        var reader = new YdbResourcesReader(ResourceInternerImpl.INSTANCE, objectMapper);
        var toKey = YdbResourcesTable.key(cloudId);
        return retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(1, TimeUnit.MINUTES)
                    .orderedRead(true)
                    .toKeyInclusive(toKey);

            if (reader.last != null) {
                settings.fromKeyExclusive(YdbResourcesTable.key(reader.last));
            } else {
                settings.fromKeyInclusive(toKey);
            }

            return session.readTable(tablePath, settings.build(), reader);
        }).thenApply(status -> {
            status.expect("can't read " + tablePath + " for " + cloudId);
            return reader.resources;
        });
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

}
