package ru.yandex.solomon.name.resolver.db.ydb;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.collection.Nullables;

import static com.yandex.ydb.table.values.PrimitiveValue.json;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbResourcesTable {
    static final StructType RESOURCE_TYPE = StructType.of(map());

    private static Map<String, Type> map() {
        var result = new HashMap<>(Map.of("hash", PrimitiveType.uint32(),
                "cloudId", PrimitiveType.utf8(),
                "resourceId", PrimitiveType.utf8(),
                "type", PrimitiveType.utf8(),
                "folderId", PrimitiveType.utf8(),
                "service", PrimitiveType.utf8(),
                "name", PrimitiveType.utf8(),
                "updatedAt", PrimitiveType.timestamp(),
                "deletedAt", PrimitiveType.timestamp().makeOptional(),
                "reindexAt", PrimitiveType.timestamp().makeOptional()));
        result.put("payload", PrimitiveType.json());
        return ImmutableMap.copyOf(result);
    }

    static final ListType RESOURCES_LIST_TYPE = ListType.of(RESOURCE_TYPE);

    static final StructType RESOURCE_PK_TYPE = StructType.of(Map.of(
            "hash", PrimitiveType.uint32(),
            "cloudId", PrimitiveType.utf8(),
            "resourceId", PrimitiveType.utf8()
    ));

    static final ListType RESOURCES_PK_LIST_TYPE = ListType.of(RESOURCE_PK_TYPE);

    static TupleValue key(Resource resource) {
        return TupleValue.of(
                uint32(resource.cloudId.hashCode()).makeOptional(),
                utf8(resource.cloudId).makeOptional(),
                utf8(resource.resourceId).makeOptional());
    }

    static Params keyParams(String cloudId) {
        return Params.create()
                .put("$hash", uint32(cloudId.hashCode()))
                .put("$cloudId", utf8(cloudId));
    }

    static TupleValue key(String cloudId) {
        return TupleValue.of(
                uint32(cloudId.hashCode()).makeOptional(),
                utf8(cloudId).makeOptional());
    }

    static List<Value> resourcesToValues(Collection<Resource> resources, ObjectMapper objectMapper) {
        ArrayList<Value> values = new ArrayList<>(resources.size());
        for (var resource : resources) {
            var map = new HashMap<>(Map.of(
                    "hash", uint32(resource.cloudId.hashCode()),
                    "cloudId", utf8(resource.cloudId),
                    "resourceId", utf8(resource.resourceId),
                    "type", utf8(resource.type),
                    "folderId", utf8(resource.folderId),
                    "service", utf8(resource.service),
                    "name", utf8(resource.name),
                    "updatedAt", timestamp(Instant.ofEpochMilli(resource.updatedAt)),
                    "deletedAt", optTimestamp(resource.deletedAt),
                    "reindexAt", optTimestamp(resource.reindexAt)
            ));
            var data = toJson(objectMapper, resource);
            map.put("payload", json(data));
            values.add(RESOURCE_TYPE.newValue(map));
        }
        return values;
    }

    static Value optTimestamp(long value) {
        if (value == 0) {
            return PrimitiveType.timestamp().makeOptional().emptyValue();
        } else {
            return timestamp(Instant.ofEpochMilli(value));
        }
    }

    static ListValue resourcesToList(Collection<Resource> resources, ObjectMapper objectMapper) {
        return RESOURCES_LIST_TYPE.newValue(resourcesToValues(resources, objectMapper));
    }

    static List<Value> resourcesToPkValues(Collection<Resource> resources) {
        var values = new ArrayList<Value>(resources.size());
        for (var resource : resources) {
            values.add(RESOURCE_PK_TYPE.newValue(Map.of(
                    "hash", uint32(resource.cloudId.hashCode()),
                    "cloudId", utf8(resource.cloudId),
                    "resourceId", utf8(resource.resourceId))
            ));
        }
        return values;
    }

    static ListValue resourcesToPkList(Collection<Resource> resources) {
        return RESOURCES_PK_LIST_TYPE.newValue(resourcesToPkValues(resources));
    }

    static TableDescription description() {
        return TableDescription.newBuilder()
                .addNullableColumn("hash", PrimitiveType.uint32())
                .addNullableColumn("cloudId", PrimitiveType.utf8())
                .addNullableColumn("resourceId", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("service", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("updatedAt", PrimitiveType.timestamp())
                .addNullableColumn("deletedAt", PrimitiveType.timestamp())
                .addNullableColumn("reindexAt", PrimitiveType.timestamp())
                .addNullableColumn("payload", PrimitiveType.json())
                .setPrimaryKeys("hash", "cloudId", "resourceId")
                .build();
    }

    static class ColumnReader {
        private final int cloudIdIdx;
        private final int resourceIdIdx;
        private final int typeIdx;
        private final int folderIdIdx;
        private final int serviceIdx;
        private final int nameIdx;
        private final int updatedAtIdx;
        private final int deletedAtIdx;
        private final int reindexAtIdx;
        private final int payloadIdx;
        private final ObjectMapper objectMapper;

        public ColumnReader(ResultSetReader resultSet, ObjectMapper objectMapper) {
            this.cloudIdIdx = resultSet.getColumnIndex("cloudId");
            this.resourceIdIdx = resultSet.getColumnIndex("resourceId");
            this.typeIdx = resultSet.getColumnIndex("type");
            this.folderIdIdx = resultSet.getColumnIndex("folderId");
            this.serviceIdx = resultSet.getColumnIndex("service");
            this.nameIdx = resultSet.getColumnIndex("name");
            this.updatedAtIdx = resultSet.getColumnIndex("updatedAt");
            this.deletedAtIdx = resultSet.getColumnIndex("deletedAt");
            this.reindexAtIdx = resultSet.getColumnIndex("reindexAt");
            this.payloadIdx = resultSet.getColumnIndex("payload");
            this.objectMapper = objectMapper;
        }

        public void read(Resource r, ResultSetReader resultSet) throws IOException {
            r.cloudId = resultSet.getColumn(cloudIdIdx).getUtf8();
            r.resourceId = resultSet.getColumn(resourceIdIdx).getUtf8();
            r.type = resultSet.getColumn(typeIdx).getUtf8();
            r.folderId = resultSet.getColumn(folderIdIdx).getUtf8();
            r.service = resultSet.getColumn(serviceIdx).getUtf8();
            r.name = resultSet.getColumn(nameIdx).getUtf8();
            r.updatedAt = resultSet.getColumn(updatedAtIdx).getTimestamp().toEpochMilli();
            r.deletedAt = resultSet.getColumn(deletedAtIdx).getTimestamp().toEpochMilli();
            r.reindexAt = resultSet.getColumn(reindexAtIdx).getTimestamp().toEpochMilli();
            String json = resultSet.getColumn(payloadIdx).getJson();
            if (json == null || json.isBlank()) {
                r.resourceComplexId = Map.of();
                r.severity = Resource.Severity.HIGHLY_CRITICAL;
                r.responsible = "";
                r.environment = "";
            } else {
                final Resource resource = objectMapper.readValue(json, Resource.class);
                r.resourceComplexId = resource.resourceComplexId;
                r.severity = resource.severity;
                if (r.severity == Resource.Severity.UNKNOWN) {
                    r.severity = Resource.Severity.HIGHLY_CRITICAL;
                }
                r.responsible = Nullables.orEmpty(resource.responsible);
                r.environment = Nullables.orEmpty(resource.environment);
            }
        }
    }
}
