package ru.yandex.solomon.name.resolver.db.ydb;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.name.resolver.db.ShardsDao;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbShardsDao implements ShardsDao {
    private final String root;
    private final String tablePath;
    private final ShardsTable table;
    private final SchemeClient scheme;

    private final String insertQuery;
    private final String deleteQuery;

    public YdbShardsDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.root = root;
        this.tablePath = root + "/Shards";
        this.table = new ShardsTable(tableClient, tablePath);
        this.scheme = schemeClient;

        this.insertQuery = String.format("""
                --!syntax_v1
                DECLARE $cloudId AS Utf8;
                REPLACE INTO `%s` (cloudId)
                VALUES ($cloudId);
                """, tablePath);

        this.deleteQuery = String.format("""
                --!syntax_v1
                DECLARE $cloudId AS Utf8;
                DELETE FROM `%s`
                WHERE cloudId = $cloudId;
                """, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return scheme.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> scheme.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Void> insert(String shardId) {
        try {
            Params params = Params.of("$cloudId", utf8(shardId));
            return table.queryVoid(insertQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> delete(String shardId) {
        try {
            Params params = Params.of("$cloudId", utf8(shardId));
            return table.queryVoid(deleteQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<String>> findAll() {
        return table.queryAll();
    }

    /**
     * SHARDS TABLE
     */
    private static final class ShardsTable extends YdbTable<String, String> {

        ShardsTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("cloudId", PrimitiveType.utf8())
                    .setPrimaryKeys("cloudId")
                    .build();
        }

        @Override
        protected String getId(String record) {
            return record;
        }

        @Override
        protected Params toParams(String record) {
            return Params.create()
                    .put("$cloudId", utf8(record));
        }

        @Override
        protected String mapFull(ResultSetReader r) {
            return r.getColumn("cloudId").getUtf8();
        }

        @Override
        protected String mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
