package ru.yandex.solomon.name.resolver.handler;

import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.grpc.utils.StreamObservers;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.name.resolver.client.grpc.Proto;
import ru.yandex.solomon.name.resolver.client.grpc.ServerStatusResponse;
import ru.yandex.solomon.name.resolver.protobuf.ResourceServiceGrpc;
import ru.yandex.solomon.name.resolver.protobuf.ServerStatusRequest;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class ServerStatusHandler {
    private static final int MAX_MESSAGE_SIZE = 10 << 20; // 10 Mib
    private final BalancerHolder balancerHolder;
    private final ClusterDiscovery<GrpcTransport> discovery;

    public ServerStatusHandler(BalancerHolder balancerHolder, ClusterDiscovery<GrpcTransport> discovery) {
        this.balancerHolder = balancerHolder;
        this.discovery = discovery;
    }

    public void serverStatus(ServerStatusRequest request, StreamObserver<ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse> observer) {
        var balancer = balancerHolder.getBalancer();
        if (balancer != null) {
            localSeverStatus(request, observer, balancer);
        } else {
            proxyServerStatus(request, observer);
        }
    }

    private void proxyServerStatus(ServerStatusRequest request, StreamObserver<ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse> observer) {
        String leader = balancerHolder.getLeaderNode();
        if (Strings.isNullOrEmpty(leader)) {
            observer.onError(Status.FAILED_PRECONDITION.withDescription("unknown leader").asRuntimeException());
            return;
        }

        var transport = discovery.getTransportByNode(leader);
        long expiredAt = request.getExpiredAt();
        var subscriber = StreamObservers.toFlowSubscriber(observer);
        transport.serverStreamingCall(ResourceServiceGrpc.getServerStatusMethod(), request, subscriber, expiredAt);
    }

    private void localSeverStatus(ServerStatusRequest request, StreamObserver<ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse> observer, Balancer balancer) {
        var seqNo = balancer.getLatestSeqNo();
        if (seqNo.getAssignSeqNo() != 0 && seqNo.getAssignSeqNo() == request.getStateHash()) {
            observer.onNext(ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse.newBuilder()
                    .setStateHash(request.getStateHash())
                    .setLeader(HostUtils.getFqdn())
                    .build());
            observer.onCompleted();
            return;
        }

        assignmentsSnapshot(balancer)
                .thenAccept(snapshot -> {
                    var messages = Proto.split(Proto.toProto(snapshot), MAX_MESSAGE_SIZE);
                    for (var message : messages) {
                        observer.onNext(message);
                    }
                })
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        observer.onError(e);
                    } else {
                        observer.onCompleted();
                    }
                });
    }

    private CompletableFuture<ServerStatusResponse> assignmentsSnapshot(Balancer balancer) {
        return balancer.supplyAsync(() -> {
            var result = new ServerStatusResponse(balancer.getLatestSeqNo().getAssignSeqNo(), HostUtils.getFqdn());
            for (var shard : balancer.getShards().values()) {
                var node = shard.getNode() != null ? shard.getNode().getAddress() : "";
                result.shardsByNode.put(node, shard.getShardId());
            }
            return result;
        });
    }
}
