package ru.yandex.solomon.name.resolver.index;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.google.common.collect.Iterators;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.search.SearchEngine;
import ru.yandex.solomon.search.SearchIndex;


/**
 * @author Sergey Polovko
 */
@Immutable
final class Level implements Iterable<Resource>, MemMeasurable {

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Level.class);
    private static final SearchEngine SEARCH_ENGINE = SearchEngine.defaultSearchEngine;
    private static final SearchIndex EMPTY_SEARCH_INDEX = SEARCH_ENGINE.build(Collections.emptyList());

    private final Map<String, Resource> resourceById;
    private final Set<String> removedResourceIds;
    private final Resource[] resources;
    private final SearchIndex searchIndex;

    Level() {
        this(new Resource[0], Map.of(), Set.of(), EMPTY_SEARCH_INDEX);
    }

    Level(Iterable<Resource> resources) {
        Map<String, Resource> resourceById = new HashMap<>();
        for (var resource : resources) {
            resourceById.put(resource.resourceId, resource);
        }
        this.resources = resourceById.values().toArray(new Resource[0]);
        this.resourceById = resourceById;
        this.removedResourceIds = Set.of();
        this.searchIndex = newIndex(this.resources);
    }

    Level(Resource[] resources, Map<String, Resource> resourceById, Set<String> removedResourceIds, SearchIndex searchIndex) {
        this.resources = resources;
        this.resourceById = resourceById;
        this.removedResourceIds = removedResourceIds;
        this.searchIndex = searchIndex;
    }

    @Nullable
    Resource getResourceById(String resourceId) {
        if (removedResourceIds.contains(resourceId)) {
            return null;
        }

        return resourceById.get(resourceId);
    }

    boolean has(String resourceId) {
        return resourceById.containsKey(resourceId) || removedResourceIds.contains(resourceId);
    }

    int size() {
        return resources.length;
    }

    // takes ownership of newResources
    Level update(Iterable<Resource> newResources) {
        var it = newResources.iterator();
        if (!it.hasNext()) {
            return this;
        }

        // (1) create new resourcesById index
        Set<String> removed = new HashSet<>(removedResourceIds);
        Map<String, Resource> resourceById = new HashMap<>(this.resourceById);
        do {
            var resource = it.next();
            resourceById.put(resource.resourceId, resource);
            removed.remove(resource.resourceId);
        } while (it.hasNext());
        var resources = resourceById.values().toArray(new Resource[0]);

        // (2) rebuild search index
        SearchIndex searchIndex = newIndex(resources);

        // (3) create new level
        return new Level(resources, resourceById, removed, searchIndex);
    }

    Level remove(List<RemoveRequest> requests) {
        Set<String> removed = new HashSet<>(removedResourceIds);
        for (var req : requests) {
            removed.addAll(req.getResourceIds());
        }
        return new Level(resources, resourceById, removed, searchIndex);
    }

    Level mergeWith(Level level, boolean latestLevel) {
        if (size() == 0) {
            return level;
        } else if (level.size() == 0) {
            return this;
        }

        try {
            // (1) merge resources
            var removed = new HashSet<>(removedResourceIds);
            removed.addAll(level.removedResourceIds);

            var resourceById = new HashMap<>(this.resourceById);
            resourceById.putAll(level.resourceById);
            resourceById.keySet().removeAll(removed);
            var resources = resourceById.values().toArray(new Resource[0]);

            // (2) rebuild search index
            SearchIndex searchIndex = newIndex(resources);

            // (3) create new level
            return new Level(resources, resourceById, latestLevel ? Set.of() : removed, searchIndex);
        } catch (Throwable t) {
            throw new RuntimeException("cannot merge levels", t);
        }
    }

    Stream<Resource> search(Selectors selectors) {
        Stream<Resource> result;
        if (selectors.isEmpty()) {
            result = resourceById.values().stream();
        } else {
            result = searchIndex.search(selectors)
                    .mapToObj(this::mapIdxToResource);
        }

        return result.filter(resource -> !removedResourceIds.contains(resource.resourceId));
    }

    private Resource mapIdxToResource(int idx) {
        return resources[idx];
    }

    private static SearchIndex newIndex(Resource[] resources) {
        var it = new ResourceLabelsIterator(Arrays.asList(resources).iterator());
        return SEARCH_ENGINE.build(it);
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += searchIndex.memorySizeIncludingSelf();
        size += MemoryCounter.arrayObjectSize(resources);
        size += MemoryCounter.hashMapSize(resourceById);
        size += MemoryCounter.hashSetSize(removedResourceIds);
        return size;
    }

    @Override
    public String toString() {
        return "Level{size: " + size() + "}";
    }

    @Override
    public Iterator<Resource> iterator() {
        var it = resourceById.values().iterator();
        if (removedResourceIds.isEmpty()) {
            return it;
        }

        return Iterators.filter(it, resource -> !removedResourceIds.contains(resource.resourceId));
    }
}
