package ru.yandex.solomon.name.resolver.index;

import java.util.Iterator;
import java.util.List;
import java.util.stream.Stream;

import javax.annotation.concurrent.Immutable;

import com.google.common.collect.Iterables;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.name.resolver.client.Resource;


/**
 * @author Sergey Polovko
 */
@Immutable
final class LevelsArray implements Iterable<Resource>, MemMeasurable {

    static final LevelsArray EMPTY = new LevelsArray(new int[0], new Level[0]);

    static final int[] DEFAULT_LEVELS_SIZES = {
        1000,               // level0    1k
        10000,              // level1    10k
        100000,             // level2    100k
        Integer.MAX_VALUE,  // level3    inf
    };

    private final int[] levelsSizes;
    private final Level[] levels;

    private LevelsArray(int[] levelsSizes, Level[] levels) {
        this.levelsSizes = levelsSizes;
        this.levels = levels;
    }

    LevelsArray(Iterable<Resource> resources) {
        this(DEFAULT_LEVELS_SIZES, resources);
    }

    LevelsArray(int[] levelsSizes, Iterable<Resource> resources) {
        final int lastLevel = levelsSizes.length - 1;

        // last level must never be overfull
        levelsSizes[lastLevel] = Integer.MAX_VALUE;
        this.levelsSizes = levelsSizes;

        // init levels (last level can be non empty from the begin)
        final Level[] levels = new Level[levelsSizes.length];
        for (int i = 0; i < lastLevel; i++) {
            levels[i] = new Level();
        }
        levels[lastLevel] = new Level(resources);
        this.levels = levels;
    }

    // takes ownership of newResources
    LevelsArray updateZeroLevel(Iterable<Resource> newResources) {
        if (levels.length == 0 || Iterables.isEmpty(newResources)) {
            return this;
        }

        Level[] newLevels = new Level[levels.length];
        newLevels[0] = levels[0].update(newResources);
        return retainOtherLevels(newLevels);
    }

    boolean has(String resourceId) {
        for (int l = levels.length - 1; l >= 0; l--) {
            if (levels[l].has(resourceId)) {
                return true;
            }
        }
        return false;
    }

    LevelsArray remove(List<RemoveRequest> requests) {
        if (levels.length == 0 || requests.isEmpty()) {
            return this;
        }

        Level[] newLevels = new Level[levels.length];
        newLevels[0] = levels[0].remove(requests);
        return retainOtherLevels(newLevels);
    }

    LevelsArray mergeOverfullLevel() {
        if (levels.length < 2) {
            return this;
        }

        // start from the bottom level to avoid stuck on merging only smaller levels
        int overfullLevelIdx = -1;
        for (int i = levels.length - 1; i >= 0; i--) {
            if (levels[i].size() > levelsSizes[i]) {
                overfullLevelIdx = i;
                break;
            }
        }

        if (overfullLevelIdx == -1) {
            return this;
        }

        // merge only one level at a time to release current thread for other work

        Level overfullLevel = levels[overfullLevelIdx];
        Level nextLevel = levels[overfullLevelIdx + 1];
        boolean last = nextLevel == levels[levels.length - 1];

        Level[] newLevels = new Level[levels.length];
        newLevels[overfullLevelIdx] = new Level();
        newLevels[overfullLevelIdx + 1] = nextLevel.mergeWith(overfullLevel, last);
        return retainOtherLevels(newLevels);
    }

    Resource getOrNull(String resourceId) {
        for (var level : levels) {
            if (level.has(resourceId)) {
                return level.getResourceById(resourceId);
            }
        }
        return null;
    }

    private LevelsArray retainOtherLevels(Level[] newLevels) {
        assert newLevels.length == levels.length;

        // fill empty array items by retaining current levels
        for (int i = 0; i < levels.length; i++) {
            if (newLevels[i] == null) {
                newLevels[i] = levels[i];
            }
        }
        return new LevelsArray(levelsSizes, newLevels);
    }

    int size() {
        int size = 0;
        for (Level level : levels) {
            size += level.size();
        }
        return size;
    }

    /**
     * @return levels count
     */
    int count() {
        return levels.length;
    }

    /**
     * @return max size of level
     */
    int levelMaxSize(int i) {
        return levelsSizes[i];
    }

    /**
     * @return size (resource count) of level
     */
    int levelSize(int i) {
        return levels[i].size();
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = 0;
        for (Level level : levels) {
            size += level.memorySizeIncludingSelf();
        }
        return size;
    }

    Stream<Resource> search(Selectors selectors) {
        var levels = this.levels;
        Stream<Resource> result = Stream.empty();
        for (int index = 0; index < levels.length; index++) {
            final int levelIdx = index;
            var stream = levels[levelIdx].search(selectors)
                    .filter(resource -> !hasOnPrevLevels(resource, levelIdx));

            result = Stream.concat(result, stream);
        }
        return result;
    }

    private boolean hasOnPrevLevels(Resource resource, int levelIdx) {
        for (int index = 0; index < levelIdx; index++) {
            if (levels[index].has(resource.resourceId)) {
                return true;
            }
        }
        return false;
    }

    Level getLevel(int levelIdx) {
        return levels[levelIdx];
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("LevelsArray{ ");
        for (int i = 0; i < levels.length; i++) {
            Level level = levels[i];
            if (level == null) {
                continue;
            }

            sb.append("  ").append("level").append(i);
            sb.append(":{ size: ").append(level.size()).append(", max: ");

            int maxSize = levelsSizes[i];
            if (maxSize == Integer.MAX_VALUE) {
                sb.append("inf");
            } else {
                sb.append(maxSize);
            }
            sb.append(" }, ");
        }
        sb.setLength(sb.length() - 2);
        sb.append(" }");
        return sb.toString();
    }

    @Override
    public Iterator<Resource> iterator() {
        return new LevelsArrayIterator(this);
    }
}
