package ru.yandex.solomon.name.resolver.index;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.solomon.name.resolver.client.Resource;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceNameIndex {
    private final Map<Key, State> stateByName = new HashMap<>();

    /**
     * @return resource that was marked as replaced
     */
    @Nullable
    public Resource add(Resource resource) {
        if (!resource.hasName()) {
            return null;
        }

        var key = Key.of(resource);
        var state = stateByName.get(key);
        if (state == null) {
            stateByName.put(key, new State(resource));
            return null;
        }

        return state.add(resource);
    }

    /**
     * @return resource that was unmark as replaced
     */
    @Nullable
    public Resource remove(@Nullable Resource resource) {
        if (resource == null || !resource.hasName()) {
            return null;
        }

        var key = Key.of(resource);
        var state = stateByName.get(key);
        if (state == null) {
            return null;
        }

        var restored = state.remove(resource);
        if (state.isEmpty()) {
            stateByName.remove(key);
        }

        return restored;
    }

    /**
     * @return resource that was unmark as a result of rename
     */
    public Resource remove(@Nullable Resource prev, Resource actual) {
        if (prev == null || !prev.hasName()) {
            return null;
        }

        var prevKey = Key.of(prev);
        var key = Key.of(actual);
        if (key.equals(prevKey)) {
            return null;
        }

        return remove(prev);
    }


    private static record Key(String folderId, String service, String type, String name) {
        public static Key of(Resource resource) {
            return new Key(resource.folderId, resource.service, resource.type, resource.name);
        }
    }

    private static class State {
        private Resource actual;
        private final Map<String, Resource> replaced = new HashMap<>();

        public State(Resource actual) {
            this.actual = actual;
        }

        @Nullable
        public Resource add(Resource resource) {
            if (actual.resourceId.equals(resource.resourceId)) {
                // update with more fresh version
                actual = resource;
                return null;
            } else if (actual.updatedAt <= resource.updatedAt) {
                return changeActual(resource);
            } else {
                var replaced = new Resource(resource).setReplaced(true);
                addReplaced(replaced);
                return replaced;
            }
        }

        public Resource remove(Resource resource) {
            if (actual.resourceId.equals(resource.resourceId)) {
                var restored = replaced.values().stream()
                        .max(Comparator.comparingLong(o -> o.updatedAt))
                        .map(r -> new Resource(r).setReplaced(false))
                        .orElse(null);
                actual = restored;
                return restored;
            }

            replaced.remove(resource.resourceId);
            return null;
        }

        public boolean isEmpty() {
            return actual == null;
        }

        private Resource changeActual(Resource resource) {
            var replace = new Resource(actual).setReplaced(true);
            addReplaced(replace);
            actual = resource;
            return replace;
        }

        private void addReplaced(Resource resource) {
            replaced.put(resource.resourceId, resource);
        }
    }
}
