package ru.yandex.solomon.name.resolver.index;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueueMem;


/**
 * Collection of resources which allowed to lookup resources by labels or search them by selectors using FTS.
 * This collection designed to be wait-free for reads and lock-free for updates.
 *
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class ResourcesCollectionImpl implements ResourcesCollection {
    private static final Logger logger = LoggerFactory.getLogger(ResourcesCollectionImpl.class);
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ResourcesCollectionImpl.class);

    private final String uniqueId;
    private final ActorRunner actor;
    private final Executor executor;
    private final ArrayListLockQueueMem<UpdateRequest> updates = new ArrayListLockQueueMem<>(2);
    private final ArrayListLockQueueMem<RemoveRequest> removals = new ArrayListLockQueueMem<>(2);

    private final ResourceNameIndex nameIndex = new ResourceNameIndex();
    private volatile LevelsArray levels;
    private volatile boolean stop = false;

    /**
     * used for debug through manager WebUI
     */
    @Nullable
    private Resource lastCreatedResource;

    // takes ownership of resources
    public ResourcesCollectionImpl(String uniqueId, Executor executor, List<Resource> resources) {
        this(uniqueId, executor, LevelsArray.DEFAULT_LEVELS_SIZES, resources);
    }

    // takes ownership of resources
    public ResourcesCollectionImpl(String uniqueId, Executor executor, int[] levelsSizes, List<Resource> resources) {
        this.uniqueId = uniqueId;
        this.actor = new ActorRunner(this::act, executor);
        this.executor = executor;
        var internedResource = resources.stream().map(ResourceInternerImpl.INSTANCE::intern).collect(Collectors.toList());
        var replaced = replaceNames(internedResource);
        this.levels = new LevelsArray(levelsSizes, Iterables.concat(internedResource, replaced));
    }

    protected void act() {
        if (stop) {
            ArrayList<UpdateRequest> updates = this.updates.dequeueAll();
            ArrayList<RemoveRequest> removals = this.removals.dequeueAll();

            LevelsArray levels = this.levels;
            if (levels != LevelsArray.EMPTY) {
                this.levels = LevelsArray.EMPTY;
            }

            if (!updates.isEmpty()) {
                completeInDifferentThread(updates);
            }

            if (!removals.isEmpty()) {
                completeInDifferentThread(removals);
            }
        } else {
            WhatThreadDoes.withNr(uniqueId + " process updates", this::processUpdates);
            WhatThreadDoes.withNr(uniqueId + " process removals", this::processRemovals);
            WhatThreadDoes.withNr(uniqueId + " process merges", this::processMerges);
        }
    }

    private void processUpdates() {
        final List<UpdateRequest> updates = this.updates.dequeueAll();
        if (updates.isEmpty()) {
            return;
        }

        WhatThreadDoes.Handle h = WhatThreadDoes.push("processUpdates in " + uniqueId);
        try {
            final LevelsArray oldLevels = levels;
            final List<Resource> newResources = mergeUpdates(updates);
            if (!newResources.isEmpty()) {
                lastCreatedResource = newResources.get(0);
                var restored = restoreNames(newResources);
                var replaced = replaceNames(newResources);
                var diff = Iterables.concat(restored, newResources, replaced);
                LevelsArray newLevels = oldLevels.updateZeroLevel(diff);
                if (newLevels != oldLevels) {
                    this.levels = newLevels;
                }
            }

            // avoid blocking current thread
            completeInDifferentThread(updates);
        } catch (Throwable t) {
            logger.error("error while processUpdates in " + uniqueId, t);
            failInDifferentThread(updates, t);
        } finally {
            h.pop();
        }
    }

    private void processRemovals() {
        final List<RemoveRequest> removals = this.removals.dequeueAll();
        if (removals.isEmpty()) {
            return;
        }

        WhatThreadDoes.Handle h = WhatThreadDoes.push("processRemovals in " + uniqueId);
        try {
            LevelsArray oldLevels = this.levels;
            LevelsArray newLevels;
            var resourceIds = removals.stream()
                    .flatMap(removeRequest -> removeRequest.getResourceIds().stream())
                    .collect(Collectors.toList());
            newLevels = oldLevels.updateZeroLevel(restoreNames(resourceIds));
            newLevels = newLevels.remove(removals);
            if (newLevels != oldLevels) {
                this.levels = newLevels;
            }

            // avoid blocking current thread
            completeInDifferentThread(removals);
        } catch (Throwable t) {
            logger.error("error while processRemovals in " + uniqueId, t);
            failInDifferentThread(removals, t);
        } finally {
            h.pop();
        }
    }

    private static List<Resource> mergeUpdates(List<UpdateRequest> updates) {
        return updates.stream()
                .flatMap(updateRequest -> updateRequest.getResources().stream())
                .map(ResourceInternerImpl.INSTANCE::intern)
                .collect(Collectors.toList());
    }

    private void processMerges() {
        WhatThreadDoes.Handle h = WhatThreadDoes.push("processMerges in " + uniqueId);
        try {
            LevelsArray oldLevels = this.levels;
            LevelsArray newLevels = oldLevels.mergeOverfullLevel();
            if (newLevels != oldLevels) {
                this.levels = newLevels;
            }
        } catch (Throwable t) {
            logger.error("error while processMerges in " + uniqueId, t);
        } finally {
            h.pop();
        }
    }

    private <C extends CompletableFuture<Void>> void completeInDifferentThread(List<C> cfs) {
        executor.execute(() -> {
            for (C cf : cfs) {
                cf.complete(null);
            }
        });
    }

    private <C extends CompletableFuture<Void>> void failInDifferentThread(List<C> cfs, Throwable t) {
        executor.execute(() -> {
            for (C cf : cfs) {
                cf.completeExceptionally(t);
            }
        });
    }

    private List<Resource> restoreNames(List<Resource> resources) {
        List<Resource> changes = new ArrayList<>();
        for (var resource : resources) {
            var prev = getOrNull(resource.resourceId);
            var restored = nameIndex.remove(prev, resource);
            if (restored != null) {
                changes.add(restored);
            }
        }
        return changes;
    }

    private List<Resource> restoreNames(Collection<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return List.of();
        }

        List<Resource> changes = new ArrayList<>();
        for (var resourceId : resourceIds) {
            var restored = nameIndex.remove(getOrNull(resourceId));
            if (restored != null) {
                changes.add(restored);
            }
        }

        return changes;
    }

    private List<Resource> replaceNames(List<Resource> resources) {
        List<Resource> changes = new ArrayList<>(resources.size());
        for (var resource : resources) {
            var replaced = nameIndex.add(resource);
            if (replaced != null) {
                changes.add(replaced);
            }
        }

        return changes;
    }

    @Override
    public Resource getOrNull(String resourceId) {
        return levels.getOrNull(resourceId);
    }

    @Override
    public boolean has(String resourceId) {
        return levels.has(resourceId);
    }

    @Override
    public CompletableFuture<Void> put(Resource resource) {
        UpdateRequest update = new UpdateRequest(resource);
        updates.enqueue(update);
        actor.schedule();
        return update;
    }

    @Override
    public CompletableFuture<Void> putAll(Collection<Resource> resources) {
        UpdateRequest update = new UpdateRequest(resources);
        updates.enqueue(update);
        actor.schedule();
        return update;
    }

    @Override
    public CompletableFuture<Void> removeAll(Collection<String> resourceIds) {
        RemoveRequest remove = new RemoveRequest(resourceIds);
        removals.enqueue(remove);
        actor.schedule();
        return remove;
    }

    @Override
    public Stream<Resource> search(Selectors selectors) {
        return levels.search(selectors);
    }

    @Override
    public int size() {
        return levels.size();
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;

        size += updates.memorySizeIncludingSelf();
        size += levels.memorySizeIncludingSelf();
        return size;
    }

    @Override
    public String toString() {
        var levels = this.levels;
        StringBuilder sb = new StringBuilder();
        sb.append("FileMetricsCollection{\n");
        for (int i = 0; i < levels.count(); i++) {
            sb.append("  ").append("level").append(i);
            sb.append(" { size: ").append(levels.levelSize(i)).append(", max: ");
            int maxSize = levels.levelMaxSize(i);
            if (maxSize == Integer.MAX_VALUE) {
                sb.append("inf");
            } else {
                sb.append(maxSize);
            }
            sb.append(" }\n");
        }
        sb.append("}");
        return sb.toString();
    }

    @Override
    public Iterator<Resource> iterator() {
        return levels.iterator();
    }

    @Override
    public void close() {
        stop = true;
        actor.schedule();
    }

    public int levelSize(int i) {
        return levels.levelSize(i);
    }
}
