package ru.yandex.solomon.name.resolver.logbroker;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import io.netty.util.AbstractReferenceCounted;
import io.netty.util.ReferenceCounted;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.persqueue.read.PartitionStreamKey;
import ru.yandex.persqueue.read.event.Message;

/**
 * @author Vladimir Gordiychuk
 */
public class PartitionStreamMetrics {
    private final String prefix;
    private final MetricRegistry registry;
    private final ConcurrentMap<PartitionStreamKey, MetricsAggr> aggregateByKey = new ConcurrentHashMap<>();
    private final ConcurrentMap<Labels, MetricsImpl> metricsByLabels = new ConcurrentHashMap<>();
    private final MetricsImpl total;

    public PartitionStreamMetrics(String prefix, MetricRegistry registry) {
        this.prefix = prefix;
        this.registry = registry;
        this.total = get(labels("total", "total", "total"));
    }

    public long uncommittedMessage() {
        return total.uncommitted.get();
    }

    public Metrics get(PartitionStreamKey key) {
        var result = aggregateByKey.get(key);
        if (result == null) {
            result = makeAggregate(key);
            aggregateByKey.put(key, result);
        }

        return result;
    }

    public void remove(PartitionStreamKey key) {
        var prev = aggregateByKey.remove(key);
        if (prev != null) {
            release(prev);
        }
    }

    public void removeAll() {
        var it = aggregateByKey.values().iterator();
        while (it.hasNext()) {
            var aggr = it.next();
            it.remove();
            release(aggr);
        }
    }

    private void release(MetricsAggr aggr) {
        for (var metrics : aggr.metrics) {
            if (metrics.release()) {
                remove(metrics.labels);
            }
        }
    }

    private MetricsAggr makeAggregate(PartitionStreamKey key) {
        MetricsImpl[] metrics = {
                get(labels(key.topic, key.cluster,Long.toUnsignedString(key.partition))),
                get(labels(key.topic, key.cluster, "total")),
                get(labels(key.topic, "total", "total")),
                get(labels("total", "total", "total"))
        };

        return new MetricsAggr(metrics);
    }

    private Labels labels(String topic, String origin, String partition) {
        return Labels.of("topic", topic, "origin", origin, "partition", partition);
    }

    private MetricsImpl get(Labels labels) {
        var result = metricsByLabels.get(labels);
        if (result == null || result.refCnt() == 0) {
            result = new MetricsImpl(prefix, labels, registry);
            metricsByLabels.put(labels, result);
            return result;
        } else {
            return (MetricsImpl) result.retain();
        }
    }

    public void remove(Labels labels) {
        var prev = metricsByLabels.get(labels);
        if (prev == null) {
            return;
        }

        if (prev.refCnt() != 0) {
            throw new IllegalStateException("Not released yet: " + labels);
        }

        metricsByLabels.remove(labels, prev);
    }

    public interface Metrics {
        void receiveMessage(Message message);
        void invalidResource();
        void filteredResource();
        void acceptedResource();
        void failed();
        void commit();
    }

    private static class MetricsImpl extends AbstractReferenceCounted {
        private final MetricRegistry root;
        private final Labels labels;
        private final Rate inboundBytes;
        private final Rate inboundMessage;
        private final Rate committedMessage;
        private final Rate failedMessage;
        private final Rate invalidResource;
        private final Rate filteredResource;
        private final Rate acceptedResource;
        private final LazyGaugeInt64 uncommitted;
        private final Histogram receiveMessageLag;

        public MetricsImpl(String prefix, Labels labels, MetricRegistry root) {
            this.labels = labels;
            this.root = root;
            var registry = root.subRegistry(labels);
            this.inboundBytes = registry.rate(prefix + ".inboundBytes");
            this.inboundMessage = registry.rate(prefix + ".inboundMessage");
            this.committedMessage = registry.rate(prefix + ".committedMessage");
            this.failedMessage = registry.rate(prefix + ".failedMessage");
            this.invalidResource = registry.rate(prefix + ".invalidResource");
            this.filteredResource = registry.rate(prefix + ".filteredResource");
            this.acceptedResource = registry.rate(prefix + ".acceptedResource");
            this.uncommitted = registry.lazyGaugeInt64(prefix + ".uncommittedMessage", () -> {
                return inboundMessage.get() - committedMessage.get() - failedMessage.get();
            });
            this.receiveMessageLag = registry.histogramRate(prefix + ".receiveMessageLagMs", Histograms.exponential(20, 2));
        }

        @Override
        protected void deallocate() {
            // remove it completely
            root.removeSubRegistry(labels);
        }

        @Override
        public ReferenceCounted touch(Object hint) {
            return this;
        }
    }

    private static class MetricsAggr implements Metrics {
        private final MetricsImpl[] metrics;

        public MetricsAggr(MetricsImpl[] metrics) {
            this.metrics = metrics;
        }

        @Override
        public void receiveMessage(Message message) {
            int bytes = message.getData().size();
            long receiveLag = System.currentTimeMillis() - message.getCreateTimeMillis();
            for (var metric : metrics) {
                metric.inboundBytes.add(bytes);
                metric.inboundMessage.inc();
                metric.receiveMessageLag.record(receiveLag);
            }
        }

        @Override
        public void invalidResource() {
            for (var metric : metrics) {
                metric.invalidResource.inc();
            }
        }

        @Override
        public void filteredResource() {
            for (var metric : metrics) {
                metric.filteredResource.inc();
            }
        }

        @Override
        public void acceptedResource() {
            for (var metric : metrics) {
                metric.acceptedResource.inc();
            }
        }

        @Override
        public void failed() {
            for (var metric : metrics) {
                metric.failedMessage.inc();
            }
        }

        @Override
        public void commit() {
            for (var metric : metrics) {
                metric.committedMessage.inc();
            }
        }
    }
}
