package ru.yandex.solomon.name.resolver.logbroker;

import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelOption;

import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.persqueue.PersqueueClient;
import ru.yandex.persqueue.settings.GrpcTransportSettings;
import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.TLogbrokerClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.ydb.YdbAuthProviders;

/**
 * @author Vladimir Gordiychuk
 */
public class PersqueueClientFactory {
    public static PersqueueClient makeClient(
            YdbAuthProviders ydbAuthProviders,
            ThreadPoolProvider threads,
            MetricRegistry registry,
            TLogbrokerClientConfig config)
    {
        var settings = GrpcTransportSettings.newBuilder()
                .endpoint(config.getEndpoint())
                .database(config.getDatabase());

        var readTimeout = TimeConverter.protoToDuration(config.getReadTimeout());
        if (!readTimeout.isZero()) {
            settings.readTimeout(readTimeout);
        }

        var callExecutor = threads.getExecutorService(
                config.getThreadPoolName(),
                "LogbrokerClient.ThreadPoolName");

        if (callExecutor != null) {
            settings.callExecutor(callExecutor);
        }

        if (config.hasTvmAuth()) {
            settings.authProvider(ydbAuthProviders.tvm(config.getTvmAuth()));
        } else if (config.hasIamKeyAuth()) {
            settings.authProvider(ydbAuthProviders.iam(config.getIamKeyAuth()));
        } else if (config.hasIamKeyJson()) {
            settings.authProvider(ydbAuthProviders.iamKeyJson(config.getIamKeyJson()));
        }

        if (config.getUseTLS()) {
            settings.secureConnection();
        }

        settings.channelInitializer(channel -> {
            channel.offloadExecutor(callExecutor);
            channel.channelType(NettyUtils.clientChannelTypeForEventLoop(threads.getIOExecutor()));
            channel.eventLoopGroup(threads.getIOExecutor());
            channel.maxInboundMessageSize(DataSizeConverter.toBytesInt(config.getMaxInboundMessageSize()));
            channel.withOption(ChannelOption.ALLOCATOR, ByteBufAllocator.DEFAULT);
            channel.withOption(ChannelOption.TCP_NODELAY, Boolean.TRUE);
            channel.withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS,
                    Math.toIntExact(TimeConverter.protoToDuration(config.getConnectTimeout()).toMillis()));
            channel.withOption(ChannelOption.SO_SNDBUF, 10 << 20); // 10 MiB
            channel.withOption(ChannelOption.SO_RCVBUF, 10 << 20); // 10 MiB
            channel.flowControlWindow(10 << 20); // 10 MiB
            channel.keepAliveTime(1, TimeUnit.MINUTES);
            channel.keepAliveTimeout(30, TimeUnit.SECONDS);
            channel.keepAliveWithoutCalls(true);
            channel.intercept(new MetricClientInterceptor("solomon", registry));
        });
        return PersqueueClient.create(settings.build());
    }
}
